/******************************************************************************
  Filename:       ustv_Communication.c
  Revised:        $Date: 2012-11-19 18:52:21 $
  Revision:       $Revision: 00000 $

  Description:   USTV functions for USTV Communication (UART Routine,
  USTV Parser, ...)

  Copyright 2012 Universit� du Sud Toulon Var. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer .You may not use this
  Software unless you agree to abide by the terms of the License. Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED ?AS IS? WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact DPT GEII at www.univ-tln.fr.

 *****************************************************************************/
/*******************************************************************************
 * INCLUDES
 ******************************************************************************/
#include "UTLN_Communication.h"
#include "Define.h"
#include "app_communication_task.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "FreeRTOS.h"
#include "event_groups.h"
#include "peripheral/gpio/plib_gpio.h"
#include "app.h"
#include "app_sd_tasks.h"
#include "HighBlue_Header.h"
#include "app_usb.h"
#include "stream_buffer.h"

/***************************************************************************************************
 * CONSTANTS
 ***************************************************************************************************/

/*******************************************************************************
 * VARIABLES
 ******************************************************************************/
extern volatile unsigned long g_longTimeStamp;
extern APP_COMMUNICATION_TASK_DATA app_communication_taskData;
extern TaskHandle_t xTaskAppHandle;
extern TaskHandle_t xTaskCNNDetectionHandle;
extern TaskHandle_t xTaskUSBHandle;
extern APP_DATA appData;
extern APP_USB_DATA app_usbData;
/*******************************************************************************
 * FONCTIONS
 ******************************************************************************/
unsigned char UartCalculateChecksum(unsigned int msgFunction,
        unsigned int msgPayloadLength, unsigned char * msgPayload)
{
    unsigned char checksum = 0;
    unsigned int i = 0;
    checksum ^= 0xFE;
    checksum ^= (unsigned char) (msgFunction >> 8);
    checksum ^= (unsigned char) (msgFunction);
    checksum ^= (unsigned char) (msgPayloadLength >> 8);
    checksum ^= (unsigned char) (msgPayloadLength);
    for (i = 0; i < msgPayloadLength; i++)
    {
        checksum ^= msgPayload[i];
    }
    return checksum;
}

//unsigned int UART3CircularRxBufferHead = 0;
//unsigned int UART3CircularRxBufferTail = 0;
volatile unsigned int UART3CircularTxBufferHead = 0;
volatile unsigned int UART3CircularTxBufferTail = 0;
//unsigned char UART3RxBuffer[UART3_CIRCULAR_BUFFER_SIZE];
volatile unsigned char UART3TxBuffer[UART3_CIRCULAR_BUFFER_SIZE];    
volatile bool UART3TxTransmissionEnCours=false;

void UART3WriteToTxBuffer(unsigned char value)
{
    UART3TxBuffer[UART3CircularTxBufferHead] = value;
    if(UART3CircularTxBufferHead<UART3_CIRCULAR_BUFFER_SIZE-1)
        UART3CircularTxBufferHead+=1;
    else
        UART3CircularTxBufferHead=0;

    //Une fois l'criture effectue, il faut transmettre vers l'UART
    //Si le buffer n'est pas en cours de vidage, on dclenche la transmission manuellement
     if(UART3IsTransmissionActive()==false)
     {
         UART3WriteFromTxBuffer();
     }
}

bool UART3IsDataReadyInTxBuffer(void)
{
    if(UART3CircularTxBufferHead != UART3CircularTxBufferTail)
        return true;
    else
        return false;
}

bool UART3IsTransmissionActive(void)
{
    return UART3TxTransmissionEnCours;
}
void UART3SetTransmissionActive(void)
{
    UART3TxTransmissionEnCours=true;
}

void UART3ClearTransmissionActive(void)
{
    UART3TxTransmissionEnCours=false;
}
void UART3WriteFromTxBuffer(void)
{
    //On passe la tranmission en tat actif
    UART3TxTransmissionEnCours = true;

    /* Initiate the transfer by writing as many bytes as we can */
    while((!(U1STA & _U1STA_UTXBF_MASK)) && UART3IsDataReadyInTxBuffer() )
    {
        unsigned char data = UART3TxBuffer[UART3CircularTxBufferTail];
        if(UART3CircularTxBufferTail<UART3_CIRCULAR_BUFFER_SIZE-1)
            UART3CircularTxBufferTail+=1;
        else
            UART3CircularTxBufferTail=0;
        U1TXREG = data;
    }
    IEC3SET = _IEC3_U1TXIE_MASK;
}

unsigned int UART3GetNbOfValuesInTxBuffer(void)
{
    if(UART3CircularTxBufferHead>=UART3CircularTxBufferTail)
        return UART3CircularTxBufferHead-UART3CircularTxBufferTail;
    else
        return UART3_CIRCULAR_BUFFER_SIZE - UART3CircularTxBufferTail + UART3CircularTxBufferHead;
}


unsigned int UART3GetRemainingSpaceInTxBuffer(void)
{
    return UART3_CIRCULAR_BUFFER_SIZE - UART3GetNbOfValuesInTxBuffer();
}


RECEPTION_STATE rx3ReceptionState = RECEPTION_WAIT;
volatile unsigned int rx3ReceivedFunction;
volatile unsigned int rx3ReceivedPayloadLength;
volatile unsigned char rx3ReceivedPayload[UART3_RX_PAYLOAD_SIZE];
volatile unsigned int rx3ReceivedPayloadIndex = 0;
//unsigned char c;
unsigned int rx3ErrorCount=0;
extern TaskHandle_t xTaskAppCommunicationHandle;

void Uart3DecodeMessage(unsigned char c)
{    
//    //Si des donn�es sont disponibles sur le port s�rie
//    while(UART3IsDataReadyInRxBuffer()==TRUE)
//    {
//        //On lit un caract�re
//        c = UART3ReadFromRxBuffer();

    switch (rx3ReceptionState)
    {
        case RECEPTION_WAIT:
            if (c == 0xFE)
                rx3ReceptionState = RECEPTION_FUNCTION_MSB;
            break;
        case RECEPTION_FUNCTION_MSB:
            rx3ReceivedFunction = (unsigned int) (c << 8);
            rx3ReceptionState = RECEPTION_FUNCTION_LSB;
            break;
        case RECEPTION_FUNCTION_LSB:
            rx3ReceivedFunction += (unsigned int) c;
            rx3ReceptionState = RECEPTION_PAYLOAD_LENGTH_MSB;
            break;
        case RECEPTION_PAYLOAD_LENGTH_MSB:
            rx3ReceivedPayloadLength = (unsigned int) (c << 8);
            rx3ReceptionState = RECEPTION_PAYLOAD_LENGTH_LSB;
            break;
        case RECEPTION_PAYLOAD_LENGTH_LSB:
            rx3ReceivedPayloadLength += (unsigned int) c;
            if (rx3ReceivedPayloadLength > UART3_RX_PAYLOAD_SIZE)
                rx3ReceptionState = RECEPTION_WAIT;
            else if (rx3ReceivedPayloadLength == 0)
                rx3ReceptionState = RECEPTION_CHECKSUM;
            else
                rx3ReceptionState = RECEPTION_PAYLOAD;
            break;
        case RECEPTION_PAYLOAD:
            rx3ReceivedPayload[rx3ReceivedPayloadIndex] = c;
            rx3ReceivedPayloadIndex++;
            if (rx3ReceivedPayloadIndex == rx3ReceivedPayloadLength)
            {
                rx3ReceivedPayloadIndex = 0;
                rx3ReceptionState = RECEPTION_CHECKSUM;
            }
            break;
        case RECEPTION_CHECKSUM:
            if (c == UartCalculateChecksum(rx3ReceivedFunction,
                    rx3ReceivedPayloadLength, (unsigned char*)rx3ReceivedPayload))
            {
                //Message valide
                if((rx3ReceivedFunction&0x1000) == 0x1000)
                {
                    //On recois un acknoledge d'un message envoy
                    /* Set bit 8 in the notification value of the task referenced by xTask1Handle. */
                    xTaskNotify( xTaskAppHandle, ACK_RECEIVED, eSetBits );
                    
                    if((rx3ReceivedFunction&0x1500)== 0x1500)
                    {
                        /* Set bit 8 in the notification value of the task referenced by xTaskCNNDetectionHandle. */
                        xTaskNotify( xTaskCNNDetectionHandle, ACK_RECEIVED, eSetBits );
                    }
                }
                else if( (rx3ReceivedFunction&0xff00) == 0x0A00)
                {
                    unsigned char outPayload[rx3ReceivedPayloadLength+UTLN_FRAME_SIZE];
                    int i;
                    outPayload[0]=SOF;
                    outPayload[1]=MSB_UINT16(rx3ReceivedFunction);
                    outPayload[2]=LSB_UINT16(rx3ReceivedFunction);
                    outPayload[3]=MSB_UINT16(rx3ReceivedPayloadLength);
                    outPayload[4]=LSB_UINT16(rx3ReceivedPayloadLength);
                    for(i=0;i<rx3ReceivedPayloadLength;i++)
                    {
                        outPayload[5+i]=rx3ReceivedPayload[i];
                    }
                    outPayload[5+rx3ReceivedPayloadLength]=c;
                    //On injecte le message dans le buffer des datas additionnelles
                    USTV_AddDataInAdditionnalBuffer(outPayload, rx3ReceivedPayloadLength+UTLN_FRAME_SIZE);
                }
                else
                {
                    //On process le message
                    ProcessMessage( rx3ReceivedFunction, rx3ReceivedPayloadLength, (unsigned char*)rx3ReceivedPayload);
                }
            }
            else
            {
                rx3ErrorCount++;
            }
            rx3ReceptionState = RECEPTION_WAIT;
            break;
        default:
            rx3ReceptionState = RECEPTION_WAIT;
            break;
    }
//    }
}





void ProcessMessage( unsigned short int command, unsigned short int length, unsigned char payload[])
{
    unsigned char blockMessage = 0;
    unsigned short int msgTxUARTPayloadLength;
    static unsigned char msgTxUARTPayload[BUFFER_TX_UART_SIZE];

    //Valeur par d�faut pour �viter de renvoyer la payload pr�c�dente si la valeur n'est pas renseign�e
    msgTxUARTPayloadLength = 0;

    switch(command)
    {        
        // GET_STATUS: demande du statut du recorder
        case GET_STATUS:
            //On prepare la reponse
            blockMessage=0;
            msgTxUARTPayload[0]=appData.status;
            //On enqueue une future command
            APP_COMMUNICATION_SetNextMessage(SEND_STATUS, 1,msgTxUARTPayload);
            break;
        
        //SET_RECORDER_PARAMETERS: configuration du recorder audio
        case SUPERVISOR_STATE:
        {
            if(length>=37)
            {
                appData.recordParameters.samplingFrequency=BUILD_UINT32(payload[0],payload[1],payload[2],payload[3]);
                appData.recordParameters.nbChannelsUsed=payload[4];
                appData.recordParameters.resolution=payload[5];
                appData.recordParameters.filterSelection=payload[6];
                appData.recordParameters.fileSizeLimitation=BUILD_UINT32(payload[7],payload[8],payload[9],payload[10]);
                appData.recordParameters.saveMode=payload[11];
                //Date courante
                appData.localTime.year=payload[12];
                appData.localTime.month=payload[13];
                appData.localTime.day=payload[14];
                appData.localTime.hour=payload[15];
                appData.localTime.min=payload[16];
                appData.localTime.sec=payload[17];
                //Date future a laquelle va demarrer l'enregistrement
                appData.timeForFileName.year=payload[18];
                appData.timeForFileName.month=payload[19];
                appData.timeForFileName.day=payload[20];
                appData.timeForFileName.hour=payload[21];
                appData.timeForFileName.min=payload[22];
                appData.timeForFileName.sec=payload[23];
                appData.supervisorStatus=payload[24];
                g_longTimeStamp=BUILD_UINT32(payload[25],payload[26],payload[27],payload[28]);
                int i;
                for(i=0;i<8;i++)
                appData.UUID64[i]=payload[29+i];
                
                if(appData.recordParameters.saveMode==REC_USB_DEV && app_usbData.currentRole==APP_HOST)
                {
                    /* Set bit 8 in the notification value of the task referenced by xTask1Handle. */
                    xTaskNotify( xTaskUSBHandle, APP_USB_SWITCHROLE_DEVICE_EVENT, eSetBits );
                    xTaskNotify( xTaskAppHandle, APP_SWITCHROLE_DEVICE_EVENT, eSetBits );
                }
                else if(appData.recordParameters.saveMode==REC_SD_HDD && app_usbData.currentRole==APP_DEVICE)
                {
                    /* Set bit 8 in the notification value of the task referenced by xTask1Handle. */
                    xTaskNotify( xTaskUSBHandle, APP_USB_SWITCHROLE_HOST_EVENT, eSetBits );                    
                }
            
                switch(appData.supervisorStatus)
                {
                    case PREPARE_RECORDING:
                    /* Set bit 8 in the notification value of the task referenced by xTask1Handle. */
                    xTaskNotify( xTaskAppHandle, PREPARE_FILE, eSetBits );
                    break;
                    case RECORDING:
                    /* Set bit 8 in the notification value of the task referenced by xTask1Handle. */
                    xTaskNotify( xTaskAppHandle, COM_START_RECORDING, eSetBits );
                    break;
                    case STOPPING:
                    /* Set bit 8 in the notification value of the task referenced by xTask1Handle. */
                    xTaskNotify( xTaskAppHandle, COM_STOP_RECORDING, eSetBits );
                    break;
                    case ANALYSE_CNN:
                        //On notifie a la tache APP que l'on fait une analyse sur le dernier fichier enregistr
                        xTaskNotify( xTaskAppHandle, COM_ANALYSE_CNN, eSetBits );
                    break;
                    case SEND_DATA_TO_GSM:
                        //On notifie a la tache APP que l'on souhaite envoyer des datas sur le canal GSM
                        xTaskNotify( xTaskAppHandle, COM_SEND_DATA_TO_GSM, eSetBits );
                    break;                
                    default:
                        break;
                }
            }
            blockMessage=1;
        }
            break; 
        case FILE_GETINFO_RESP:

            
            break;

 
        //SET_EXTERNAL_PERIPHERALS: Configuration des priphriques externes
        //Commande speciale, egalement forwardee dans le buffer additionnel
        case SET_EXTERNAL_PERIPHERALS:
        {
            unsigned char peripheralCount=MIN(payload[0], MAX_PERIPHERAL);
            PERIPHERAL_CONFIGURATION config[peripheralCount];
            int i;
            for(i=0;i<peripheralCount;i++)
            {
                config[i].Type=payload[i*6+1];
                config[i].ID=payload[i*6+2];
                config[i].Range=payload[i*6+3];
                config[i].Resolution=payload[i*6+4];
                config[i].Frequency= BUILD_UINT16(payload[i*6+5],payload[i*6+6]);
                appData.peripheralConfig[i]=config[i];
            }
            //On injecte la trame dans le buffer des donnes additionnelles
            MakeAndAddMessageWithUTLNProtocolInAdditionnalBuffer(command, length, payload);
            blockMessage=0;
            break;
        }
        //START_RECORDING: Demarre l'enregistrement    
        case START_RECORDING:
            /* Set bit 8 in the notification value of the task referenced by xTask1Handle. */
            xTaskNotify( xTaskAppHandle, COM_START_RECORDING, eSetBits );
            blockMessage=0;
            break;
        //STOP_RECORDING: Arrete l'enregistrement    
        case STOP_RECORDING:
            /* Set bit 8 in the notification value of the task referenced by xTask1Handle. */
            xTaskNotify( xTaskAppHandle, COM_STOP_RECORDING, eSetBits );
            blockMessage=0;
            break;
        //TOA_EVENT_FRAME: Arrete l'enregistrement    
        case TOA_EVENT_FRAME:
        {
            uint8_t espece=payload[0];
            if(espece)
            {
                //On a detecter un Rorqual
                //On notifie a la tache APP qu'une detection a eu lieu, celle-ci doit demarrer un enregistrement
                //puis doit lancer une analyse CNN
                /* Set bit 10 in the notification value of the task referenced by xTask1Handle. */
                xTaskNotify( xTaskAppHandle, PRE_DETECTION_RORQUAL, eSetBits );
            }
            else
            {
                //On a detecter un Cachalot
                //On notifie a la tache APP qu'une detection a eu lieu, celle-ci doit demarrer un enregistrement
                //puis doit lancer une analyse CNN
                /* Set bit 11 in the notification value of the task referenced by xTask1Handle. */
                xTaskNotify( xTaskAppHandle, PRE_DETECTION_CACHALOT, eSetBits );
            }
            blockMessage=0;
        }
        break;       
        // Unknown command
        default:
        {
            blockMessage=1;
            msgTxUARTPayloadLength = 1;
            msgTxUARTPayload[0]=(unsigned char)0;
        }
        break;
    }
    //On envoie le message
    if(blockMessage==0)
    {
        MakeAndSendMessageWithUTLNProtocol(ACKNOLEDGE(command), msgTxUARTPayloadLength, msgTxUARTPayload);
    }
    blockMessage=0;
}


extern StreamBufferHandle_t xStreamBufferUart1Tx;
void MakeAndSendMessageWithUTLNProtocol(unsigned short command, unsigned int payloadLength, unsigned char* payload)
{
    //portENTER_CRITICAL();
    unsigned char outPayload[payloadLength+UTLN_FRAME_SIZE];
    int i;
    outPayload[0]=SOF;
    outPayload[1]=MSB_UINT16(command);
    outPayload[2]=LSB_UINT16(command);
    outPayload[3]=MSB_UINT16(payloadLength);
    outPayload[4]=LSB_UINT16(payloadLength);
    for(i=0;i<payloadLength;i++)
    {
        outPayload[5+i]=payload[i];
    }
    outPayload[5+payloadLength]=UartCalculateChecksum(command,payloadLength, payload);
    
    //SendMessageToUart1( UTLN_FRAME_SIZE+payloadLength, outPayload);
    if(UART3GetRemainingSpaceInTxBuffer()>payloadLength+5)
    {
        int i;
        for(i=0;i<payloadLength+6;i++)
        {
            UART3WriteToTxBuffer(outPayload[i]);
        }
    }

//    size_t xBytesSent=0;
////    BaseType_t xHigherPriorityTaskWoken = pdFALSE; /* Initialised to pdFALSE. */
///* Attempt to send the string to the stream buffer. */
//    xBytesSent = xStreamBufferSend( xStreamBufferUart1Tx,
//                                           ( void * ) outPayload,
//                                          UTLN_FRAME_SIZE+payloadLength,
//                                          0 );
//    if(xBytesSent)
//        ;
    //portEXIT_CRITICAL();
}


void SendMessageToUart1( unsigned short int payloadLength, unsigned char* payload)
{
    //DRV_USART_WriteBufferAdd(app_communication_taskData.usart1Handle,payload,payloadLength,&app_communication_taskData.writeBufferHandle);
    
    if(UART3GetRemainingSpaceInTxBuffer()>payloadLength)
    {
        int i;
        for(i=0;i<payloadLength;i++)
        {
            UART3WriteToTxBuffer(payload[i]);
        }
    }
}

//============================================================================//
void MakeAndAddMessageWithUTLNProtocolInAdditionnalBuffer(unsigned short command, unsigned int payloadLength, unsigned char* payload)
{
    unsigned char outPayload[payloadLength+UTLN_FRAME_SIZE];
    int i;
    outPayload[0]=SOF;
    outPayload[1]=MSB_UINT16(command);
    outPayload[2]=LSB_UINT16(command);
    outPayload[3]=MSB_UINT16(payloadLength);
    outPayload[4]=LSB_UINT16(payloadLength);
    for(i=0;i<payloadLength;i++)
    {
        outPayload[5+i]=payload[i];
    }
    outPayload[5+payloadLength]=UartCalculateChecksum(command,payloadLength, payload);
    
    USTV_AddDataInAdditionnalBuffer(outPayload, UTLN_FRAME_SIZE+payloadLength);
}
/*******************************************************************************
 End of File
*/


