/*******************************************************************************
  MPLAB Harmony Application Source File

  Company:
    Microchip Technology Inc.

  File Name:
    app_usb.c

  Summary:
    This file contains the source code for the MPLAB Harmony application.

  Description:
    This file contains the source code for the MPLAB Harmony application.  It
    implements the logic of the application's state machine and it may call
    API routines of other MPLAB Harmony modules in the system, such as drivers,
    system services, and middleware.  However, it does not call any of the
    system interfaces (such as the "Initialize" and "Tasks" functions) of any of
    the modules in the system or make any assumptions about when those functions
    are called.  That is the responsibility of the configuration-specific system
    files.
 *******************************************************************************/

// *****************************************************************************
// *****************************************************************************
// Section: Included Files
// *****************************************************************************
// *****************************************************************************

#include "app_usb.h"
#include "config/default/peripheral/gpio/plib_gpio.h"
#include "USTV_SystemTrace.h"
#include "UTLN_Communication.h"
#include "Define.h"
// *****************************************************************************
// *****************************************************************************
// Section: Global Data Definitions
// *****************************************************************************
// *****************************************************************************
#define APP_EP_BULK_OUT 1
#define APP_EP_BULK_IN 1
// *****************************************************************************
/* Application Data

  Summary:
    Holds application data

  Description:
    This structure holds the application's data.

  Remarks:
    This structure should be initialized by the APP_USB_Initialize function.

    Application strings and buffers are be defined outside this structure.
*/

APP_USB_DATA app_usbData;

TaskHandle_t xTaskUSBHandle=NULL;
extern TaskHandle_t xTaskAppHandle;
extern APP_DATA appData;

/* The endpoint size is 64 for FS and 512 for HS */
uint16_t endpointSize;
/* Receive data buffer */
uint8_t receivedDataBuffer[512] CACHE_ALIGN;
uint8_t transmitDataBuffer[512] CACHE_ALIGN;
// *****************************************************************************
// *****************************************************************************
// Section: Application Callback Functions
// *****************************************************************************
// *****************************************************************************

/*******************************************************************************
  Function:
    void APP_USBDeviceEventHandler (USB_DEVICE_EVENT event,
        USB_DEVICE_EVENT_DATA * eventData)

  Summary:
    Event callback generated by USB device layer.

  Description:
    This event handler will handle all device layer events.

  Parameters:
    None.

  Returns:
    None.
*/

void APP_USBDeviceEventHandler(USB_DEVICE_EVENT event, void * eventData, uintptr_t context)
{
    USB_DEVICE_EVENT_DATA_CONFIGURED * configurationValue;
    USB_SETUP_PACKET * setupPacket;
    switch(event)
    {
        case USB_DEVICE_EVENT_SOF:
            /* This event is used for switch debounce. This flag is reset
             * by the switch process routine. */
            break;
        case USB_DEVICE_EVENT_RESET:
        case USB_DEVICE_EVENT_DECONFIGURED:
        
            /* Device got deconfigured */
            app_usbData.isConfigured = false;
            LED_VERTE_Clear();
//            app_usbData.state = APP_STATE_WAIT_FOR_CONFIGURATION;
            break;

        case USB_DEVICE_EVENT_CONFIGURED:

            /* Device is configured */
            configurationValue = (USB_DEVICE_EVENT_DATA_CONFIGURED *)eventData;
            if(configurationValue->configurationValue == 1)
            {
                app_usbData.isConfigured = true;
                LED_VERTE_Set();
              
                /* Register the Application HID Event Handler. */
//                USB_DEVICE_HID_EventHandlerSet(app_usbData.hidInstance,
//                        APP_USBDeviceHIDEventHandler, (uintptr_t)&appData);
            }
            break;

        case USB_DEVICE_EVENT_POWER_DETECTED:
            app_usbData.powerDetected = true;
            break;

        case USB_DEVICE_EVENT_POWER_REMOVED:
            app_usbData.powerDetected = false;

            break;
        case USB_DEVICE_EVENT_CONTROL_TRANSFER_SETUP_REQUEST:
            /* This means we have received a setup packet */
            setupPacket = (USB_SETUP_PACKET *)eventData;
            if(setupPacket->bRequest == USB_REQUEST_SET_INTERFACE)
            {
                /* If we have got the SET_INTERFACE request, we just acknowledge
                 for now. This demo has only one alternate setting which is already
                 active. */
                USB_DEVICE_ControlStatus(app_usbData.deviceHandle,USB_DEVICE_CONTROL_STATUS_OK);
            }
            else if(setupPacket->bRequest == USB_REQUEST_GET_INTERFACE)
            {
                /* We have only one alternate setting and this setting 0. So
                 * we send this information to the host. */

                USB_DEVICE_ControlSend(app_usbData.deviceHandle, &app_usbData.altSetting, 1);
            }
            else
            {
                /* We have received a request that we cannot handle. Stall it*/
                USB_DEVICE_ControlStatus(app_usbData.deviceHandle, USB_DEVICE_CONTROL_STATUS_ERROR);
            }
            break;
        case USB_DEVICE_EVENT_SUSPENDED:
            break;
            
        case USB_DEVICE_EVENT_ENDPOINT_READ_COMPLETE:
           /* Endpoint read is complete */
            app_usbData.epDataReadPending = false;
            break;

        case USB_DEVICE_EVENT_ENDPOINT_WRITE_COMPLETE:
            /* Endpoint write is complete */
            app_usbData.epDataWritePending = false;
            LED_VERTE_Toggle();
            break;

        case USB_DEVICE_EVENT_RESUMED:
        case USB_DEVICE_EVENT_ERROR:
        default:

            break;

    } 
}

USB_HOST_EVENT_RESPONSE APP_USBHostEventHandler (USB_HOST_EVENT event, void * eventData, uintptr_t context)
{
    switch (event)
    {
        case USB_HOST_EVENT_DEVICE_REJECTED_INSUFFICIENT_POWER:
            #ifdef USE_SYSTEM_TRACE
                USTV_SystemTraceWriteLine("ERROR:USB_HOST_INSUFFICIENT_POWER");
            #endif
            LED_ROUGE_Set();
            break;
        case USB_HOST_EVENT_DEVICE_UNSUPPORTED:
            event=event;
            #ifdef USE_SYSTEM_TRACE
                USTV_SystemTraceWriteLine("ERROR:USB_HOST_DEVICE_UNSUPPORTED");
            #endif
            break;
        default:
            break;
                    
    }
    
    return(USB_HOST_EVENT_RESPONSE_NONE);
}

// *****************************************************************************
// *****************************************************************************
// Section: Application Local Functions
// *****************************************************************************
// *****************************************************************************


void USB_HOST_EN_PowerEnable(void)
{
    USB_HOST_EN_Set();
}

void USB_HOST_EN_PowerDisable(void)
{
    USB_HOST_EN_Clear();
}
bool device;
void APP_SYSFSEventHandler(SYS_FS_EVENT event, void * eventData, uintptr_t context)
{
    switch(event)
    {
        case SYS_FS_EVENT_MOUNT:
            device = true;
            break;
            
        case SYS_FS_EVENT_UNMOUNT:
            device = false;
            
            break;
            
        default:
            break;
    }
}
// *****************************************************************************
// *****************************************************************************
// Section: Application Initialization and State Machine Functions
// *****************************************************************************
// *****************************************************************************

/*******************************************************************************
  Function:
    void APP_USB_Initialize ( void )

  Remarks:
    See prototype in app_usb.h.
 */

void APP_USB_Initialize ( void )
{
    /* Place the App state machine in its initial state. */
    app_usbData.state = APP_USB_STATE_INIT;

    app_usbData.deviceHandle  = USB_DEVICE_HANDLE_INVALID;
    app_usbData.isConfigured = false;
    app_usbData.deviceIsConnected = false;
    app_usbData.currentRole = APP_HOST ;          //Default role at boot
    app_usbData.roleSwitch = true;                  //Enable a roleswitch to init default mode
    app_usbData.powerDetected = false;
    app_usbData.endpointRx= (APP_EP_BULK_OUT | USB_EP_DIRECTION_OUT);
    app_usbData.endpointTx = (APP_EP_BULK_IN | USB_EP_DIRECTION_IN);
    
    //SYS_FS_EventHandlerSet(APP_SYSFSEventHandler, (uintptr_t)NULL);
}


void APP_USB_ChangeRole(uint8_t role)
{
    if(role!=app_usbData.currentRole)
    {
        if(role==APP_HOST)
        {
            app_usbData.currentRole = APP_HOST ;          //Default role at boot
            app_usbData.roleSwitch = true;                  //Enable a roleswitch to init default mode
        }
        else
        {
            app_usbData.currentRole = APP_DEVICE ;          //Default role at boot
            app_usbData.roleSwitch = true;                  //Enable a roleswitch to init default mode
            app_usbData.state = APP_STATE_SELECT_USB_ROLE;
        }
    }
}
/******************************************************************************
  Function:
    void APP_USB_Tasks ( void )

  Remarks:
    See prototype in app_usb.h.
 */

void APP_USB_Tasks ( void )
{
    /* Check the application's current state. */
    switch ( app_usbData.state )
    {
        /* Application's initial state. */
        case APP_USB_STATE_INIT:
        {
            xTaskUSBHandle=xTaskGetCurrentTaskHandle();
            //On register le callBack de la fonction HOST
            USB_HOST_EventHandlerSet(APP_USBHostEventHandler, 0);

            /* Open the device layer */
            app_usbData.deviceHandle = USB_DEVICE_Open( USB_DEVICE_INDEX_0,
                    DRV_IO_INTENT_READWRITE );

            if(app_usbData.deviceHandle != USB_DEVICE_HANDLE_INVALID)
            {
                /* Register a callback with device layer to get event notification (for end point 0) */
                USB_DEVICE_EventHandlerSet(app_usbData.deviceHandle,
                        APP_USBDeviceEventHandler, 0);

                app_usbData.state = APP_STATE_SELECT_USB_ROLE;
            }
            else
            {
                /* The Device Layer is not ready to be opened. We should try
                 * again later. */
            }
            break;
        }

        case APP_STATE_SELECT_USB_ROLE:
        {
            //Si on a une demande de changement de role
            if(app_usbData.roleSwitch)
            {
                app_usbData.isConfigured = false;
                app_usbData.deviceIsConnected = false;
 
                //On demande a passer en HOST
                if(app_usbData.currentRole == APP_HOST  )
                {
                    app_usbData.state = APP_STATE_WAIT_FOR_BUS_ENABLE_COMPLETE;
                    USB_HOST_BusEnable(0);
                    if ( app_usbData.powerDetected == false )
                    {
                        USB_DEVICE_Detach(app_usbData.deviceHandle);
                    }
                    app_usbData.currentRole = APP_HOST ;
                    app_usbData.roleSwitch = false;  
                }
                
                //On demande a passer en device
                if(app_usbData.currentRole == APP_DEVICE  )
                {
                    app_usbData.state = APP_STATE_WAIT_FOR_BUS_DISABLE_COMPLETE;
                    USB_HOST_BusDisable(0);
                    app_usbData.currentRole = APP_DEVICE ;
                    app_usbData.roleSwitch = false;
                }
            }
            break;
        }

        case APP_STATE_WAIT_FOR_BUS_DISABLE_COMPLETE:
            if(USB_HOST_RESULT_TRUE == USB_HOST_BusIsDisabled(0))
            {
                app_usbData.state = APP_STATE_WAIT_FOR_POWER_DETECT;
            }
            
            break;
            
        case APP_STATE_WAIT_FOR_POWER_DETECT :
            
            if ( app_usbData.powerDetected == true )
            {
                /* VBUS was detected. We can attach the device */
                USB_DEVICE_Attach(app_usbData.deviceHandle);
                app_usbData.state = APP_STATE_WAIT_FOR_CONFIGURATION;
            }
            break;

        //En attente de la configuration du mode Device par le PC (enumeration)
        case APP_STATE_WAIT_FOR_CONFIGURATION:

            /* Check if the device is configured. The 
             * isConfigured flag is updated in the
             * Device Event Handler */

            if(app_usbData.isConfigured)
            {
                if (USB_DEVICE_ActiveSpeedGet(app_usbData.deviceHandle) == USB_SPEED_FULL)
                {
                    endpointSize = 64;
                }
                else if (USB_DEVICE_ActiveSpeedGet(app_usbData.deviceHandle) == USB_SPEED_HIGH)
                {
                    endpointSize = 512;
                }
                if (USB_DEVICE_EndpointIsEnabled(app_usbData.deviceHandle, app_usbData.endpointRx) == false )
                {
                    /* Enable Read Endpoint */
                    USB_DEVICE_EndpointEnable(app_usbData.deviceHandle, 0, app_usbData.endpointRx,
                            USB_TRANSFER_TYPE_BULK, endpointSize);
                }
                if (USB_DEVICE_EndpointIsEnabled(app_usbData.deviceHandle, app_usbData.endpointTx) == false )
                {
                    /* Enable Write Endpoint */
                    USB_DEVICE_EndpointEnable(app_usbData.deviceHandle, 0, app_usbData.endpointTx,
                            USB_TRANSFER_TYPE_BULK, endpointSize);
                }
                /* Indicate that we are waiting for read */
                app_usbData.epDataReadPending = true;

                /* Place a new read request. */
                USB_DEVICE_EndpointRead(app_usbData.deviceHandle, &app_usbData.readTranferHandle,
                        app_usbData.endpointRx, &receivedDataBuffer[0], sizeof(receivedDataBuffer) );
                
                app_usbData.state = APP_STATE_RUNNING;
            }
            break;

        case APP_STATE_WAIT_FOR_BUS_ENABLE_COMPLETE:

            if( USB_HOST_RESULT_TRUE == USB_HOST_BusIsEnabled(0) )
            {
                app_usbData.state = APP_STATE_WAIT_FOR_DEVICE_ATTACH;
            }
            break;
        
        //Attente d'une connection de peripherique (HDD)
        case APP_STATE_WAIT_FOR_DEVICE_ATTACH:

            /* Wait for device attach. The state machine will move
             * to the next state when the attach event
             * is received.  */
            if(app_usbData.deviceIsConnected)
            {
                app_usbData.state = APP_STATE_DEVICE_CONNECTED;
            }
            else
            {
                //On attend une notification du bouton (call from interrupt)
                BaseType_t xResult;
                uint32_t ulNotifiedValue=0;
                 /* Wait to be notified of an interrupt. */
                xResult = xTaskNotifyWait( pdFALSE,    /* Don't clear bits on entry. */
                   0xffffffff,        /* Clear all bits on exit. */
                   &ulNotifiedValue, /* Stores the notified value. */
                   0x0 );               //Timout 0
                if( xResult == pdPASS )
                {
                   /* A notification was received.  See which bits were set. */
                   if( ( ulNotifiedValue & APP_USB_SWITCHROLE_DEVICE_EVENT ) != 0 )
                   {
                       if(app_usbData.currentRole==APP_HOST)
                        APP_USB_ChangeRole(APP_DEVICE);
                   }   
                   /* A notification was received.  See which bits were set. */
                   if( ( ulNotifiedValue & APP_USB_SWITCHROLE_HOST_EVENT ) != 0 )
                   {
                       if(app_usbData.currentRole==APP_DEVICE)
                        APP_USB_ChangeRole(APP_HOST);
                   }   
                }
            }

            break;

        //Un peripherique (HDD) a ete connecte    
        case APP_STATE_DEVICE_CONNECTED:

            /* Device was connected. We can try mounting the disk */
            app_usbData.state = APP_STATE_IDLE;
            break;
            
        case APP_STATE_IDLE:
        {
            //On attend une notification du bouton (call from interrupt)
            BaseType_t xResult;
            uint32_t ulNotifiedValue=0;
             /* Wait to be notified of an interrupt. */
            xResult = xTaskNotifyWait( pdFALSE,    /* Don't clear bits on entry. */
               0xffffffff,        /* Clear all bits on exit. */
               &ulNotifiedValue, /* Stores the notified value. */
               0xFFFFFFFF );               //Timout 100ms
            if( xResult == pdPASS )
            {
               /* A notification was received.  See which bits were set. */
               if( ( ulNotifiedValue & APP_USB_SWITCHROLE_DEVICE_EVENT ) != 0 )
               {
                   if(app_usbData.currentRole==APP_HOST)
                    APP_USB_ChangeRole(APP_DEVICE);
               }   
               /* A notification was received.  See which bits were set. */
               if( ( ulNotifiedValue & APP_USB_SWITCHROLE_HOST_EVENT ) != 0 )
               {
                   if(app_usbData.currentRole==APP_DEVICE)
                    APP_USB_ChangeRole(APP_HOST);
               }   
            }
        }
            break;
        case APP_STATE_RUNNING:
        {
            if(!app_usbData.isConfigured)
            {
                /* This means the device got deconfigured. Change the
                 * application state back to waiting for configuration. */
                app_usbData.state = APP_STATE_WAIT_FOR_CONFIGURATION;

                /* Disable the endpoint*/
                USB_DEVICE_EndpointDisable(app_usbData.deviceHandle, app_usbData.endpointRx);
                USB_DEVICE_EndpointDisable(app_usbData.deviceHandle, app_usbData.endpointTx);
                app_usbData.epDataReadPending = false;
                app_usbData.epDataWritePending = false;
            }
            else if (app_usbData.epDataReadPending == false)
            {
                /* Look at the data the host sent, to see what kind of
                 * application specific command it sent. */
                if(receivedDataBuffer[0]==0xFE)
                {
                    USBDecodeMessage(receivedDataBuffer, 128);
                }

                app_usbData.epDataReadPending = true ;

                /* Place a new read request. */
                USB_DEVICE_EndpointRead ( app_usbData.deviceHandle, &app_usbData.readTranferHandle,
                        app_usbData.endpointRx, &receivedDataBuffer[0], sizeof(receivedDataBuffer) );
            }
            
            
            if(USBIsDataReadyInTxBuffer()&& app_usbData.isConfigured && !app_usbData.epDataWritePending)
            {
                USBWriteFromTxBuffer();
            }
        }
            break;            
        /* The default state should never be executed. */
        default:
        {
            /* TODO: Handle error in application's state machine. */
            break;
        }
    }
}

RECEPTION_STATE usbReceptionState = RECEPTION_WAIT;
unsigned int usbReceivedFunction;
unsigned int usbReceivedPayloadLength;
unsigned char usbReceivedPayload[UART3_RX_PAYLOAD_SIZE];
volatile unsigned int usbReceivedPayloadIndex = 0;
//unsigned char c;
unsigned int usbErrorCount=0;

void USBDecodeMessage(unsigned char* usbPayload, unsigned int usbPayloadLength)
{    
    for(int i=0;i<usbPayloadLength;i++)
    {
        unsigned char c=usbPayload[i];
        switch (usbReceptionState)
        {
            case RECEPTION_WAIT:
                if (c == 0xFE)
                    usbReceptionState = RECEPTION_FUNCTION_MSB;
                break;
            case RECEPTION_FUNCTION_MSB:
                usbReceivedFunction = (unsigned int) (c << 8);
                usbReceptionState = RECEPTION_FUNCTION_LSB;
                break;
            case RECEPTION_FUNCTION_LSB:
                usbReceivedFunction += (unsigned int) c;
                usbReceptionState = RECEPTION_PAYLOAD_LENGTH_MSB;
                break;
            case RECEPTION_PAYLOAD_LENGTH_MSB:
                usbReceivedPayloadLength = (unsigned int) (c << 8);
                usbReceptionState = RECEPTION_PAYLOAD_LENGTH_LSB;
                break;
            case RECEPTION_PAYLOAD_LENGTH_LSB:
                usbReceivedPayloadLength += (unsigned int) c;
                if (usbReceivedPayloadLength > UART3_RX_PAYLOAD_SIZE-1)
                    usbReceptionState = RECEPTION_WAIT;
                else if (usbReceivedPayloadLength == 0)
                    usbReceptionState = RECEPTION_CHECKSUM;
                else
                    usbReceptionState = RECEPTION_PAYLOAD;
                break;
            case RECEPTION_PAYLOAD:
                usbReceivedPayload[usbReceivedPayloadIndex] = c;
                usbReceivedPayloadIndex++;
                if (usbReceivedPayloadIndex == usbReceivedPayloadLength)
                {
                    usbReceivedPayloadIndex = 0;
                    usbReceptionState = RECEPTION_CHECKSUM;
                }
                break;
            case RECEPTION_CHECKSUM:
                if (c == UartCalculateChecksum(usbReceivedFunction,
                        usbReceivedPayloadLength, (unsigned char*)usbReceivedPayload))
                {
                    //Message valide
                    //On process le message
                    //ProcessUSBMessage( usbReceivedFunction, usbReceivedPayloadLength, (unsigned char*)usbReceivedPayload);
                    usbReceivedPayload[usbReceivedPayloadLength]=c;
                    
                    MakeAndSendMessageWithUTLNProtocol(usbReceivedFunction, usbReceivedPayloadLength, usbReceivedPayload);
                    
                }
                else
                {
                    usbErrorCount++;
                }
                usbReceptionState = RECEPTION_WAIT;
                break;
            default:
                usbReceptionState = RECEPTION_WAIT;
                break;
        }
    }
}

void MakeAndSendUSBMessageWithUTLNProtocol(unsigned short command, unsigned int payloadLength, unsigned char* payload)
{
    unsigned char outPayload[payloadLength+UTLN_FRAME_SIZE];
    int i;
    outPayload[0]=SOF;
    outPayload[1]=MSB_UINT16(command);
    outPayload[2]=LSB_UINT16(command);
    outPayload[3]=MSB_UINT16(payloadLength);
    outPayload[4]=LSB_UINT16(payloadLength);
    for(i=0;i<payloadLength;i++)
    {
        outPayload[5+i]=payload[i];
    }
    outPayload[5+payloadLength]=UartCalculateChecksum(command,payloadLength, payload);
    
    for(int j=0;j<UTLN_FRAME_SIZE+payloadLength;j++)
    {
        USBWriteToTxBuffer(outPayload[j]);
    }
}

void MakeAndSendUSBMessageWithUTLNProtocol_direct(unsigned short command, unsigned int payloadLength, unsigned char* payload)
{
//    unsigned char outPayload[payloadLength+UTLN_FRAME_SIZE];
    int i;
    transmitDataBuffer[0]=SOF;
    transmitDataBuffer[1]=MSB_UINT16(command);
    transmitDataBuffer[2]=LSB_UINT16(command);
    transmitDataBuffer[3]=MSB_UINT16(payloadLength);
    transmitDataBuffer[4]=LSB_UINT16(payloadLength);
    for(i=0;i<payloadLength;i++)
    {
        transmitDataBuffer[5+i]=payload[i];
    }
    transmitDataBuffer[5+payloadLength]=UartCalculateChecksum(command,payloadLength, payload);
    
    /* Send the data to the host */
    app_usbData.epDataWritePending = true;

    USB_DEVICE_EndpointWrite ( app_usbData.deviceHandle, &app_usbData.writeTranferHandle,
            app_usbData.endpointTx, &transmitDataBuffer[0],
            UTLN_FRAME_SIZE+payloadLength,
            USB_DEVICE_TRANSFER_FLAGS_MORE_DATA_PENDING);
}

void ProcessUSBMessage( unsigned short int command, unsigned short int length, unsigned char payload[])
{
    unsigned char blockMessage = 0;
    unsigned short int msgTxUARTPayloadLength;
    static unsigned char msgTxUARTPayload[BUFFER_TX_UART_SIZE];

    //Valeur par d�faut pour �viter de renvoyer la payload pr�c�dente si la valeur n'est pas renseign�e
    msgTxUARTPayloadLength = 0;

    switch(command)
    {        
        // GET_STATUS: demande du statut du recorder
        case GET_STATUS:
            //On prepare la reponse
            blockMessage=0;
            msgTxUARTPayload[0]=appData.status;
            //On enqueue une future command
            APP_COMMUNICATION_SetNextMessage(SEND_STATUS, 1,msgTxUARTPayload);
            break;
        
        //SET_RECORDER_PARAMETERS: configuration du recorder audio
        case SUPERVISOR_STATE:
        {
            if(length>=37)
            {
                appData.recordParameters.samplingFrequency=BUILD_UINT32(payload[0],payload[1],payload[2],payload[3]);
                appData.recordParameters.nbChannelsUsed=payload[4];
                appData.recordParameters.resolution=payload[5];
                appData.recordParameters.filterSelection=payload[6];
                appData.recordParameters.fileSizeLimitation=BUILD_UINT32(payload[7],payload[8],payload[9],payload[10]);
                appData.recordParameters.saveMode=payload[11];
                //Date future a laquelle va demarrer l'enregistrement
                appData.localTime.year=payload[12];
                appData.localTime.month=payload[13];
                appData.localTime.day=payload[14];
                appData.localTime.hour=payload[15];
                appData.localTime.min=payload[16];
                appData.localTime.sec=payload[17];
                //Date future a laquelle va demarrer l'enregistrement
                appData.timeForFileName.year=payload[18];
                appData.timeForFileName.month=payload[19];
                appData.timeForFileName.day=payload[20];
                appData.timeForFileName.hour=payload[21];
                appData.timeForFileName.min=payload[22];
                appData.timeForFileName.sec=payload[23];
                appData.supervisorStatus=payload[24];
                g_longTimeStamp=BUILD_UINT32(payload[25],payload[26],payload[27],payload[28]);
                int i;
                for(i=0;i<8;i++)
                appData.UUID64[i]=payload[29+i];
            }
            
            switch(appData.supervisorStatus)
            {
                case PREPARE_RECORDING:
                /* Set bit 8 in the notification value of the task referenced by xTask1Handle. */
                xTaskNotify( xTaskAppHandle, PREPARE_FILE, eSetBits );
                break;
                case RECORDING:
                /* Set bit 8 in the notification value of the task referenced by xTask1Handle. */
                xTaskNotify( xTaskAppHandle, COM_START_RECORDING, eSetBits );
                break;
                case STOPPING:
                /* Set bit 8 in the notification value of the task referenced by xTask1Handle. */
                xTaskNotify( xTaskAppHandle, COM_STOP_RECORDING, eSetBits );
                break;
                default:
                    break;
            }
            blockMessage=1;
        }
            break;           
 
        //SET_EXTERNAL_PERIPHERALS: Configuration des priphriques externes
        //Commande speciale, egalement forwardee dans le buffer additionnel
        case SET_EXTERNAL_PERIPHERALS:
        {
            unsigned char peripheralCount=MIN(payload[0], MAX_PERIPHERAL);
            PERIPHERAL_CONFIGURATION config[peripheralCount];
            int i;
            for(i=0;i<peripheralCount;i++)
            {
                config[i].Type=payload[i*6+1];
                config[i].ID=payload[i*6+2];
                config[i].Range=payload[i*6+3];
                config[i].Resolution=payload[i*6+4];
                config[i].Frequency= BUILD_UINT16(payload[i*6+5],payload[i*6+6]);
                appData.peripheralConfig[i]=config[i];
            }
            //On injecte la trame dans le buffer des donnes additionnelles
            MakeAndAddMessageWithUTLNProtocolInAdditionnalBuffer(command, length, payload);
            blockMessage=0;
            break;
        }
        //START_RECORDING: Demarre l'enregistrement    
        case START_RECORDING:
            /* Set bit 8 in the notification value of the task referenced by xTask1Handle. */
            xTaskNotify( xTaskAppHandle, COM_START_RECORDING, eSetBits );
            blockMessage=0;
            break;
        //STOP_RECORDING: Arrete l'enregistrement    
        case STOP_RECORDING:
            /* Set bit 8 in the notification value of the task referenced by xTask1Handle. */
            xTaskNotify( xTaskAppHandle, COM_STOP_RECORDING, eSetBits );
            blockMessage=0;
            break;
    
        // Unknown command
        default:
        {
            blockMessage=1;
            msgTxUARTPayloadLength = 1;
            msgTxUARTPayload[0]=(unsigned char)0;
        }
        break;
    }
    
    //On envoie le message
    if(blockMessage==0)
    {
        MakeAndSendUSBMessageWithUTLNProtocol(ACKNOLEDGE(command), msgTxUARTPayloadLength, msgTxUARTPayload);
    }
    blockMessage=0;
}

//<editor-fold defaultstate="collapsed" desc="USB CircularBuffer">
volatile unsigned int USBCircularTxBufferHead = 0;
volatile unsigned int USBCircularTxBufferTail = 0;
volatile unsigned char USBTxBuffer[USB_CIRCULAR_BUFFER_SIZE];    
volatile bool USBTxTransmissionEnCours=false;

void USBWriteToTxBuffer(unsigned char value)
{
    USBTxBuffer[USBCircularTxBufferHead] = value;
    if(USBCircularTxBufferHead<USB_CIRCULAR_BUFFER_SIZE-1)
        USBCircularTxBufferHead+=1;
    else
        USBCircularTxBufferHead=0;
}

bool USBIsDataReadyInTxBuffer(void)
{
    if(USBCircularTxBufferHead != USBCircularTxBufferTail)
        return true;
    else
        return false;
}

void USBWriteFromTxBuffer(void)
{
    int index=0;

    /* Initiate the transfer by writing as many bytes as we can */
    while(index<512 && USBIsDataReadyInTxBuffer() )
    {
        unsigned char data = USBTxBuffer[USBCircularTxBufferTail];
        if(USBCircularTxBufferTail<USB_CIRCULAR_BUFFER_SIZE-1)
            USBCircularTxBufferTail+=1;
        else
            USBCircularTxBufferTail=0;
        transmitDataBuffer[index++] = data;
    }
    /* Send the data to the host */
    app_usbData.epDataWritePending = true;

    USB_DEVICE_EndpointWrite ( app_usbData.deviceHandle, &app_usbData.writeTranferHandle,
            app_usbData.endpointTx, &transmitDataBuffer[0],
            index,
            USB_DEVICE_TRANSFER_FLAGS_MORE_DATA_PENDING);
}

unsigned int USBGetNbOfValuesInTxBuffer(void)
{
    if(USBCircularTxBufferHead>=USBCircularTxBufferTail)
        return USBCircularTxBufferHead-USBCircularTxBufferTail;
    else
        return USB_CIRCULAR_BUFFER_SIZE - USBCircularTxBufferTail + USBCircularTxBufferHead;
}


unsigned int USBGetRemainingSpaceInTxBuffer(void)
{
    return USB_CIRCULAR_BUFFER_SIZE - USBGetNbOfValuesInTxBuffer();
}
//</editor-fold>
/*******************************************************************************
 End of File
 */
