/*******************************************************************************
  GPIO PLIB

  Company:
    Microchip Technology Inc.

  File Name:
    plib_gpio.h

  Summary:
    GPIO PLIB Header File

  Description:
    This library provides an interface to control and interact with Parallel
    Input/Output controller (GPIO) module.

*******************************************************************************/

/*******************************************************************************
* Copyright (C) 2019 Microchip Technology Inc. and its subsidiaries.
*
* Subject to your compliance with these terms, you may use Microchip software
* and any derivatives exclusively with Microchip products. It is your
* responsibility to comply with third party license terms applicable to your
* use of third party software (including open source software) that may
* accompany Microchip software.
*
* THIS SOFTWARE IS SUPPLIED BY MICROCHIP "AS IS". NO WARRANTIES, WHETHER
* EXPRESS, IMPLIED OR STATUTORY, APPLY TO THIS SOFTWARE, INCLUDING ANY IMPLIED
* WARRANTIES OF NON-INFRINGEMENT, MERCHANTABILITY, AND FITNESS FOR A
* PARTICULAR PURPOSE.
*
* IN NO EVENT WILL MICROCHIP BE LIABLE FOR ANY INDIRECT, SPECIAL, PUNITIVE,
* INCIDENTAL OR CONSEQUENTIAL LOSS, DAMAGE, COST OR EXPENSE OF ANY KIND
* WHATSOEVER RELATED TO THE SOFTWARE, HOWEVER CAUSED, EVEN IF MICROCHIP HAS
* BEEN ADVISED OF THE POSSIBILITY OR THE DAMAGES ARE FORESEEABLE. TO THE
* FULLEST EXTENT ALLOWED BY LAW, MICROCHIP'S TOTAL LIABILITY ON ALL CLAIMS IN
* ANY WAY RELATED TO THIS SOFTWARE WILL NOT EXCEED THE AMOUNT OF FEES, IF ANY,
* THAT YOU HAVE PAID DIRECTLY TO MICROCHIP FOR THIS SOFTWARE.
*******************************************************************************/

#ifndef PLIB_GPIO_H
#define PLIB_GPIO_H

#include <device.h>
#include <stdint.h>
#include <stdbool.h>
#include <stddef.h>

// DOM-IGNORE-BEGIN
#ifdef __cplusplus  // Provide C++ Compatibility

    extern "C" {

#endif
// DOM-IGNORE-END

// *****************************************************************************
// *****************************************************************************
// Section: Data types and constants
// *****************************************************************************
// *****************************************************************************


/*** Macros for ADC_OSR0 pin ***/
#define ADC_OSR0_Set()               (LATASET = (1<<9))
#define ADC_OSR0_Clear()             (LATACLR = (1<<9))
#define ADC_OSR0_Toggle()            (LATAINV= (1<<9))
#define ADC_OSR0_OutputEnable()      (TRISACLR = (1<<9))
#define ADC_OSR0_InputEnable()       (TRISASET = (1<<9))
#define ADC_OSR0_Get()               ((PORTA >> 9) & 0x1)
#define ADC_OSR0_PIN                  GPIO_PIN_RA9

/*** Macros for IO_5 pin ***/
#define IO_5_Set()               (LATBSET = (1<<15))
#define IO_5_Clear()             (LATBCLR = (1<<15))
#define IO_5_Toggle()            (LATBINV= (1<<15))
#define IO_5_OutputEnable()      (TRISBCLR = (1<<15))
#define IO_5_InputEnable()       (TRISBSET = (1<<15))
#define IO_5_Get()               ((PORTB >> 15) & 0x1)
#define IO_5_PIN                  GPIO_PIN_RB15

/*** Macros for IO_19 pin ***/
#define IO_19_Set()               (LATBSET = (1<<3))
#define IO_19_Clear()             (LATBCLR = (1<<3))
#define IO_19_Toggle()            (LATBINV= (1<<3))
#define IO_19_OutputEnable()      (TRISBCLR = (1<<3))
#define IO_19_InputEnable()       (TRISBSET = (1<<3))
#define IO_19_Get()               ((PORTB >> 3) & 0x1)
#define IO_19_PIN                  GPIO_PIN_RB3

/*** Macros for SPI1_SS pin ***/
#define SPI1_SS_Get()               ((PORTB >> 8) & 0x1)
#define SPI1_SS_PIN                  GPIO_PIN_RB8

/*** Macros for IO_7 pin ***/
#define IO_7_Set()               (LATBSET = (1<<1))
#define IO_7_Clear()             (LATBCLR = (1<<1))
#define IO_7_Toggle()            (LATBINV= (1<<1))
#define IO_7_OutputEnable()      (TRISBCLR = (1<<1))
#define IO_7_InputEnable()       (TRISBSET = (1<<1))
#define IO_7_Get()               ((PORTB >> 1) & 0x1)
#define IO_7_PIN                  GPIO_PIN_RB1

/*** Macros for IO_20 pin ***/
#define IO_20_Set()               (LATCSET = (1<<2))
#define IO_20_Clear()             (LATCCLR = (1<<2))
#define IO_20_Toggle()            (LATCINV= (1<<2))
#define IO_20_OutputEnable()      (TRISCCLR = (1<<2))
#define IO_20_InputEnable()       (TRISCSET = (1<<2))
#define IO_20_Get()               ((PORTC >> 2) & 0x1)
#define IO_20_PIN                  GPIO_PIN_RC2

/*** Macros for IO_8 pin ***/
#define IO_8_Set()               (LATCSET = (1<<3))
#define IO_8_Clear()             (LATCCLR = (1<<3))
#define IO_8_Toggle()            (LATCINV= (1<<3))
#define IO_8_OutputEnable()      (TRISCCLR = (1<<3))
#define IO_8_InputEnable()       (TRISCSET = (1<<3))
#define IO_8_Get()               ((PORTC >> 3) & 0x1)
#define IO_8_PIN                  GPIO_PIN_RC3

/*** Macros for TX_PIC32 pin ***/
#define TX_PIC32_Get()               ((PORTC >> 4) & 0x1)
#define TX_PIC32_PIN                  GPIO_PIN_RC4

/*** Macros for SPI1_MISO pin ***/
#define SPI1_MISO_Get()               ((PORTC >> 1) & 0x1)
#define SPI1_MISO_PIN                  GPIO_PIN_RC1

/*** Macros for ADC_FILTER0 pin ***/
#define ADC_FILTER0_Set()               (LATESET = (1<<7))
#define ADC_FILTER0_Clear()             (LATECLR = (1<<7))
#define ADC_FILTER0_Toggle()            (LATEINV= (1<<7))
#define ADC_FILTER0_OutputEnable()      (TRISECLR = (1<<7))
#define ADC_FILTER0_InputEnable()       (TRISESET = (1<<7))
#define ADC_FILTER0_Get()               ((PORTE >> 7) & 0x1)
#define ADC_FILTER0_PIN                  GPIO_PIN_RE7

/*** Macros for IO_9 pin ***/
#define IO_9_Set()               (LATESET = (1<<8))
#define IO_9_Clear()             (LATECLR = (1<<8))
#define IO_9_Toggle()            (LATEINV= (1<<8))
#define IO_9_OutputEnable()      (TRISECLR = (1<<8))
#define IO_9_InputEnable()       (TRISESET = (1<<8))
#define IO_9_Get()               ((PORTE >> 8) & 0x1)
#define IO_9_PIN                  GPIO_PIN_RE8

/*** Macros for IO_21 pin ***/
#define IO_21_Set()               (LATESET = (1<<9))
#define IO_21_Clear()             (LATECLR = (1<<9))
#define IO_21_Toggle()            (LATEINV= (1<<9))
#define IO_21_OutputEnable()      (TRISECLR = (1<<9))
#define IO_21_InputEnable()       (TRISESET = (1<<9))
#define IO_21_Get()               ((PORTE >> 9) & 0x1)
#define IO_21_PIN                  GPIO_PIN_RE9

/*** Macros for SPI5_SS pin ***/
#define SPI5_SS_Get()               ((PORTF >> 12) & 0x1)
#define SPI5_SS_PIN                  GPIO_PIN_RF12

/*** Macros for SPI5_CLK pin ***/
#define SPI5_CLK_Get()               ((PORTF >> 13) & 0x1)
#define SPI5_CLK_PIN                  GPIO_PIN_RF13

/*** Macros for SPI2_CLK pin ***/
#define SPI2_CLK_Get()               ((PORTG >> 6) & 0x1)
#define SPI2_CLK_PIN                  GPIO_PIN_RG6

/*** Macros for RESET_ADC pin ***/
#define RESET_ADC_Set()               (LATGSET = (1<<8))
#define RESET_ADC_Clear()             (LATGCLR = (1<<8))
#define RESET_ADC_Toggle()            (LATGINV= (1<<8))
#define RESET_ADC_OutputEnable()      (TRISGCLR = (1<<8))
#define RESET_ADC_InputEnable()       (TRISGSET = (1<<8))
#define RESET_ADC_Get()               ((PORTG >> 8) & 0x1)
#define RESET_ADC_PIN                  GPIO_PIN_RG8

/*** Macros for SPI6_MISO pin ***/
#define SPI6_MISO_Get()               ((PORTG >> 9) & 0x1)
#define SPI6_MISO_PIN                  GPIO_PIN_RG9

/*** Macros for ADC_HR pin ***/
#define ADC_HR_Set()               (LATHSET = (1<<4))
#define ADC_HR_Clear()             (LATHCLR = (1<<4))
#define ADC_HR_Toggle()            (LATHINV= (1<<4))
#define ADC_HR_OutputEnable()      (TRISHCLR = (1<<4))
#define ADC_HR_InputEnable()       (TRISHSET = (1<<4))
#define ADC_HR_Get()               ((PORTH >> 4) & 0x1)
#define ADC_HR_PIN                  GPIO_PIN_RH4

/*** Macros for USB_HOST_EN pin ***/
#define USB_HOST_EN_Set()               (LATESET = (1<<1))
#define USB_HOST_EN_Clear()             (LATECLR = (1<<1))
#define USB_HOST_EN_Toggle()            (LATEINV= (1<<1))
#define USB_HOST_EN_OutputEnable()      (TRISECLR = (1<<1))
#define USB_HOST_EN_InputEnable()       (TRISESET = (1<<1))
#define USB_HOST_EN_Get()               ((PORTE >> 1) & 0x1)
#define USB_HOST_EN_PIN                  GPIO_PIN_RE1

/*** Macros for USB_OVER_CURRENT pin ***/
#define USB_OVER_CURRENT_Set()               (LATESET = (1<<0))
#define USB_OVER_CURRENT_Clear()             (LATECLR = (1<<0))
#define USB_OVER_CURRENT_Toggle()            (LATEINV= (1<<0))
#define USB_OVER_CURRENT_OutputEnable()      (TRISECLR = (1<<0))
#define USB_OVER_CURRENT_InputEnable()       (TRISESET = (1<<0))
#define USB_OVER_CURRENT_Get()               ((PORTE >> 0) & 0x1)
#define USB_OVER_CURRENT_PIN                  GPIO_PIN_RE0

/*** Macros for IO_0 pin ***/
#define IO_0_Set()               (LATHSET = (1<<14))
#define IO_0_Clear()             (LATHCLR = (1<<14))
#define IO_0_Toggle()            (LATHINV= (1<<14))
#define IO_0_OutputEnable()      (TRISHCLR = (1<<14))
#define IO_0_InputEnable()       (TRISHSET = (1<<14))
#define IO_0_Get()               ((PORTH >> 14) & 0x1)
#define IO_0_PIN                  GPIO_PIN_RH14

/*** Macros for IO_1 pin ***/
#define IO_1_Set()               (LATESET = (1<<4))
#define IO_1_Clear()             (LATECLR = (1<<4))
#define IO_1_Toggle()            (LATEINV= (1<<4))
#define IO_1_OutputEnable()      (TRISECLR = (1<<4))
#define IO_1_InputEnable()       (TRISESET = (1<<4))
#define IO_1_Get()               ((PORTE >> 4) & 0x1)
#define IO_1_PIN                  GPIO_PIN_RE4

/*** Macros for SPI1_CLK pin ***/
#define SPI1_CLK_Get()               ((PORTD >> 1) & 0x1)
#define SPI1_CLK_PIN                  GPIO_PIN_RD1

/*** Macros for SPI5_MOSI pin ***/
#define SPI5_MOSI_Get()               ((PORTD >> 2) & 0x1)
#define SPI5_MOSI_PIN                  GPIO_PIN_RD2

/*** Macros for SPI4_SS pin ***/
#define SPI4_SS_Get()               ((PORTD >> 9) & 0x1)
#define SPI4_SS_PIN                  GPIO_PIN_RD9

/*** Macros for SPI4_CLK pin ***/
#define SPI4_CLK_Get()               ((PORTD >> 10) & 0x1)
#define SPI4_CLK_PIN                  GPIO_PIN_RD10

/*** Macros for IO_6 pin ***/
#define IO_6_Set()               (LATDSET = (1<<0))
#define IO_6_Clear()             (LATDCLR = (1<<0))
#define IO_6_Toggle()            (LATDINV= (1<<0))
#define IO_6_OutputEnable()      (TRISDCLR = (1<<0))
#define IO_6_InputEnable()       (TRISDSET = (1<<0))
#define IO_6_Get()               ((PORTD >> 0) & 0x1)
#define IO_6_PIN                  GPIO_PIN_RD0

/*** Macros for SPI3_SS pin ***/
#define SPI3_SS_Get()               ((PORTD >> 12) & 0x1)
#define SPI3_SS_PIN                  GPIO_PIN_RD12

/*** Macros for SPI3_MOSI pin ***/
#define SPI3_MOSI_Get()               ((PORTF >> 5) & 0x1)
#define SPI3_MOSI_PIN                  GPIO_PIN_RF5

/*** Macros for SPI2_MOSI pin ***/
#define SPI2_MOSI_Get()               ((PORTF >> 1) & 0x1)
#define SPI2_MOSI_PIN                  GPIO_PIN_RF1

/*** Macros for SPI1_MOSI pin ***/
#define SPI1_MOSI_Get()               ((PORTF >> 0) & 0x1)
#define SPI1_MOSI_PIN                  GPIO_PIN_RF0

/*** Macros for SPI4_MOSI pin ***/
#define SPI4_MOSI_Get()               ((PORTF >> 3) & 0x1)
#define SPI4_MOSI_PIN                  GPIO_PIN_RF3

/*** Macros for SPI3_MISO pin ***/
#define SPI3_MISO_Get()               ((PORTG >> 1) & 0x1)
#define SPI3_MISO_PIN                  GPIO_PIN_RG1

/*** Macros for SPI4_MISO pin ***/
#define SPI4_MISO_Get()               ((PORTG >> 0) & 0x1)
#define SPI4_MISO_PIN                  GPIO_PIN_RG0

/*** Macros for SPI5_MISO pin ***/
#define SPI5_MISO_Get()               ((PORTF >> 4) & 0x1)
#define SPI5_MISO_PIN                  GPIO_PIN_RF4

/*** Macros for IO_11 pin ***/
#define IO_11_Set()               (LATESET = (1<<3))
#define IO_11_Clear()             (LATECLR = (1<<3))
#define IO_11_Toggle()            (LATEINV= (1<<3))
#define IO_11_OutputEnable()      (TRISECLR = (1<<3))
#define IO_11_InputEnable()       (TRISESET = (1<<3))
#define IO_11_Get()               ((PORTE >> 3) & 0x1)
#define IO_11_PIN                  GPIO_PIN_RE3

/*** Macros for IO_4 pin ***/
#define IO_4_Set()               (LATHSET = (1<<6))
#define IO_4_Clear()             (LATHCLR = (1<<6))
#define IO_4_Toggle()            (LATHINV= (1<<6))
#define IO_4_OutputEnable()      (TRISHCLR = (1<<6))
#define IO_4_InputEnable()       (TRISHSET = (1<<6))
#define IO_4_Get()               ((PORTH >> 6) & 0x1)
#define IO_4_PIN                  GPIO_PIN_RH6

/*** Macros for ADC_START pin ***/
#define ADC_START_Set()               (LATHSET = (1<<12))
#define ADC_START_Clear()             (LATHCLR = (1<<12))
#define ADC_START_Toggle()            (LATHINV= (1<<12))
#define ADC_START_OutputEnable()      (TRISHCLR = (1<<12))
#define ADC_START_InputEnable()       (TRISHSET = (1<<12))
#define ADC_START_Get()               ((PORTH >> 12) & 0x1)
#define ADC_START_PIN                  GPIO_PIN_RH12

/*** Macros for LED_ORANGE pin ***/
#define LED_ORANGE_Set()               (LATHSET = (1<<8))
#define LED_ORANGE_Clear()             (LATHCLR = (1<<8))
#define LED_ORANGE_Toggle()            (LATHINV= (1<<8))
#define LED_ORANGE_OutputEnable()      (TRISHCLR = (1<<8))
#define LED_ORANGE_InputEnable()       (TRISHSET = (1<<8))
#define LED_ORANGE_Get()               ((PORTH >> 8) & 0x1)
#define LED_ORANGE_PIN                  GPIO_PIN_RH8

/*** Macros for LED_ROUGE pin ***/
#define LED_ROUGE_Set()               (LATHSET = (1<<1))
#define LED_ROUGE_Clear()             (LATHCLR = (1<<1))
#define LED_ROUGE_Toggle()            (LATHINV= (1<<1))
#define LED_ROUGE_OutputEnable()      (TRISHCLR = (1<<1))
#define LED_ROUGE_InputEnable()       (TRISHSET = (1<<1))
#define LED_ROUGE_Get()               ((PORTH >> 1) & 0x1)
#define LED_ROUGE_PIN                  GPIO_PIN_RH1

/*** Macros for LED_VERTE pin ***/
#define LED_VERTE_Set()               (LATHSET = (1<<0))
#define LED_VERTE_Clear()             (LATHCLR = (1<<0))
#define LED_VERTE_Toggle()            (LATHINV= (1<<0))
#define LED_VERTE_OutputEnable()      (TRISHCLR = (1<<0))
#define LED_VERTE_InputEnable()       (TRISHSET = (1<<0))
#define LED_VERTE_Get()               ((PORTH >> 0) & 0x1)
#define LED_VERTE_PIN                  GPIO_PIN_RH0

/*** Macros for SW3 pin ***/
#define SW3_Set()               (LATJSET = (1<<9))
#define SW3_Clear()             (LATJCLR = (1<<9))
#define SW3_Toggle()            (LATJINV= (1<<9))
#define SW3_OutputEnable()      (TRISJCLR = (1<<9))
#define SW3_InputEnable()       (TRISJSET = (1<<9))
#define SW3_Get()               ((PORTJ >> 9) & 0x1)
#define SW3_PIN                  GPIO_PIN_RJ9
#define SW3_InterruptEnable()   (CNENJSET = (1<<9))
#define SW3_InterruptDisable()  (CNENJCLR = (1<<9))

/*** Macros for SW2 pin ***/
#define SW2_Set()               (LATJSET = (1<<8))
#define SW2_Clear()             (LATJCLR = (1<<8))
#define SW2_Toggle()            (LATJINV= (1<<8))
#define SW2_OutputEnable()      (TRISJCLR = (1<<8))
#define SW2_InputEnable()       (TRISJSET = (1<<8))
#define SW2_Get()               ((PORTJ >> 8) & 0x1)
#define SW2_PIN                  GPIO_PIN_RJ8
#define SW2_InterruptEnable()   (CNENJSET = (1<<8))
#define SW2_InterruptDisable()  (CNENJCLR = (1<<8))

/*** Macros for SW1 pin ***/
#define SW1_Set()               (LATJSET = (1<<1))
#define SW1_Clear()             (LATJCLR = (1<<1))
#define SW1_Toggle()            (LATJINV= (1<<1))
#define SW1_OutputEnable()      (TRISJCLR = (1<<1))
#define SW1_InputEnable()       (TRISJSET = (1<<1))
#define SW1_Get()               ((PORTJ >> 1) & 0x1)
#define SW1_PIN                  GPIO_PIN_RJ1
#define SW1_InterruptEnable()   (CNENJSET = (1<<1))
#define SW1_InterruptDisable()  (CNENJCLR = (1<<1))

/*** Macros for IO_12 pin ***/
#define IO_12_Set()               (LATDSET = (1<<11))
#define IO_12_Clear()             (LATDCLR = (1<<11))
#define IO_12_Toggle()            (LATDINV= (1<<11))
#define IO_12_OutputEnable()      (TRISDCLR = (1<<11))
#define IO_12_InputEnable()       (TRISDSET = (1<<11))
#define IO_12_Get()               ((PORTD >> 11) & 0x1)
#define IO_12_PIN                  GPIO_PIN_RD11

/*** Macros for SPI2_MISO pin ***/
#define SPI2_MISO_Get()               ((PORTD >> 7) & 0x1)
#define SPI2_MISO_PIN                  GPIO_PIN_RD7

/*** Macros for SPI6_SS pin ***/
#define SPI6_SS_Get()               ((PORTD >> 6) & 0x1)
#define SPI6_SS_PIN                  GPIO_PIN_RD6

/*** Macros for IO_13 pin ***/
#define IO_13_Set()               (LATASET = (1<<15))
#define IO_13_Clear()             (LATACLR = (1<<15))
#define IO_13_Toggle()            (LATAINV= (1<<15))
#define IO_13_OutputEnable()      (TRISACLR = (1<<15))
#define IO_13_InputEnable()       (TRISASET = (1<<15))
#define IO_13_Get()               ((PORTA >> 15) & 0x1)
#define IO_13_PIN                  GPIO_PIN_RA15

/*** Macros for IO_14 pin ***/
#define IO_14_Set()               (LATASET = (1<<14))
#define IO_14_Clear()             (LATACLR = (1<<14))
#define IO_14_Toggle()            (LATAINV= (1<<14))
#define IO_14_OutputEnable()      (TRISACLR = (1<<14))
#define IO_14_InputEnable()       (TRISASET = (1<<14))
#define IO_14_Get()               ((PORTA >> 14) & 0x1)
#define IO_14_PIN                  GPIO_PIN_RA14

/*** Macros for ADC_FILTER1 pin ***/
#define ADC_FILTER1_Set()               (LATKSET = (1<<2))
#define ADC_FILTER1_Clear()             (LATKCLR = (1<<2))
#define ADC_FILTER1_Toggle()            (LATKINV= (1<<2))
#define ADC_FILTER1_OutputEnable()      (TRISKCLR = (1<<2))
#define ADC_FILTER1_InputEnable()       (TRISKSET = (1<<2))
#define ADC_FILTER1_Get()               ((PORTK >> 2) & 0x1)
#define ADC_FILTER1_PIN                  GPIO_PIN_RK2

/*** Macros for ADC_OSR1 pin ***/
#define ADC_OSR1_Set()               (LATKSET = (1<<1))
#define ADC_OSR1_Clear()             (LATKCLR = (1<<1))
#define ADC_OSR1_Toggle()            (LATKINV= (1<<1))
#define ADC_OSR1_OutputEnable()      (TRISKCLR = (1<<1))
#define ADC_OSR1_InputEnable()       (TRISKSET = (1<<1))
#define ADC_OSR1_Get()               ((PORTK >> 1) & 0x1)
#define ADC_OSR1_PIN                  GPIO_PIN_RK1

/*** Macros for IO_2 pin ***/
#define IO_2_Set()               (LATHSET = (1<<7))
#define IO_2_Clear()             (LATHCLR = (1<<7))
#define IO_2_Toggle()            (LATHINV= (1<<7))
#define IO_2_OutputEnable()      (TRISHCLR = (1<<7))
#define IO_2_InputEnable()       (TRISHSET = (1<<7))
#define IO_2_Get()               ((PORTH >> 7) & 0x1)
#define IO_2_PIN                  GPIO_PIN_RH7

/*** Macros for IO_3 pin ***/
#define IO_3_Set()               (LATHSET = (1<<3))
#define IO_3_Clear()             (LATHCLR = (1<<3))
#define IO_3_Toggle()            (LATHINV= (1<<3))
#define IO_3_OutputEnable()      (TRISHCLR = (1<<3))
#define IO_3_InputEnable()       (TRISHSET = (1<<3))
#define IO_3_Get()               ((PORTH >> 3) & 0x1)
#define IO_3_PIN                  GPIO_PIN_RH3

/*** Macros for ADC_FSMODE pin ***/
#define ADC_FSMODE_Set()               (LATJSET = (1<<2))
#define ADC_FSMODE_Clear()             (LATJCLR = (1<<2))
#define ADC_FSMODE_Toggle()            (LATJINV= (1<<2))
#define ADC_FSMODE_OutputEnable()      (TRISJCLR = (1<<2))
#define ADC_FSMODE_InputEnable()       (TRISJSET = (1<<2))
#define ADC_FSMODE_Get()               ((PORTJ >> 2) & 0x1)
#define ADC_FSMODE_PIN                  GPIO_PIN_RJ2

/*** Macros for SPI6_CLK pin ***/
#define SPI6_CLK_Get()               ((PORTD >> 15) & 0x1)
#define SPI6_CLK_PIN                  GPIO_PIN_RD15

/*** Macros for SYNCHRO_PPS pin ***/
#define SYNCHRO_PPS_Get()               ((PORTD >> 14) & 0x1)
#define SYNCHRO_PPS_PIN                  GPIO_PIN_RD14

/*** Macros for SPI3_CLK pin ***/
#define SPI3_CLK_Get()               ((PORTB >> 14) & 0x1)
#define SPI3_CLK_PIN                  GPIO_PIN_RB14

/*** Macros for IO_17 pin ***/
#define IO_17_Set()               (LATBSET = (1<<5))
#define IO_17_Clear()             (LATBCLR = (1<<5))
#define IO_17_Toggle()            (LATBINV= (1<<5))
#define IO_17_OutputEnable()      (TRISBCLR = (1<<5))
#define IO_17_InputEnable()       (TRISBSET = (1<<5))
#define IO_17_Get()               ((PORTB >> 5) & 0x1)
#define IO_17_PIN                  GPIO_PIN_RB5

/*** Macros for RX_PIC32 pin ***/
#define RX_PIC32_Get()               ((PORTB >> 10) & 0x1)
#define RX_PIC32_PIN                  GPIO_PIN_RB10

/*** Macros for SPI6_MOSI pin ***/
#define SPI6_MOSI_Get()               ((PORTB >> 0) & 0x1)
#define SPI6_MOSI_PIN                  GPIO_PIN_RB0

/*** Macros for SPI2_SS pin ***/
#define SPI2_SS_Get()               ((PORTB >> 2) & 0x1)
#define SPI2_SS_PIN                  GPIO_PIN_RB2

/*** Macros for IO_18 pin ***/
#define IO_18_Set()               (LATBSET = (1<<9))
#define IO_18_Clear()             (LATBCLR = (1<<9))
#define IO_18_Toggle()            (LATBINV= (1<<9))
#define IO_18_OutputEnable()      (TRISBCLR = (1<<9))
#define IO_18_InputEnable()       (TRISBSET = (1<<9))
#define IO_18_Get()               ((PORTB >> 9) & 0x1)
#define IO_18_PIN                  GPIO_PIN_RB9

/*** Macros for IO_10 pin ***/
#define IO_10_Set()               (LATBSET = (1<<4))
#define IO_10_Clear()             (LATBCLR = (1<<4))
#define IO_10_Toggle()            (LATBINV= (1<<4))
#define IO_10_OutputEnable()      (TRISBCLR = (1<<4))
#define IO_10_InputEnable()       (TRISBSET = (1<<4))
#define IO_10_Get()               ((PORTB >> 4) & 0x1)
#define IO_10_PIN                  GPIO_PIN_RB4


// *****************************************************************************
/* GPIO Port

  Summary:
    Identifies the available GPIO Ports.

  Description:
    This enumeration identifies the available GPIO Ports.

  Remarks:
    The caller should not rely on the specific numbers assigned to any of
    these values as they may change from one processor to the next.

    Not all ports are available on all devices.  Refer to the specific
    device data sheet to determine which ports are supported.
*/

typedef enum
{
    GPIO_PORT_A = 0,
    GPIO_PORT_B = 1,
    GPIO_PORT_C = 2,
    GPIO_PORT_D = 3,
    GPIO_PORT_E = 4,
    GPIO_PORT_F = 5,
    GPIO_PORT_G = 6,
    GPIO_PORT_H = 7,
    GPIO_PORT_J = 8,
    GPIO_PORT_K = 9,
} GPIO_PORT;

// *****************************************************************************
/* GPIO Port Pins

  Summary:
    Identifies the available GPIO port pins.

  Description:
    This enumeration identifies the available GPIO port pins.

  Remarks:
    The caller should not rely on the specific numbers assigned to any of
    these values as they may change from one processor to the next.

    Not all pins are available on all devices.  Refer to the specific
    device data sheet to determine which pins are supported.
*/

typedef enum
{
    GPIO_PIN_RA0 = 0,
    GPIO_PIN_RA1 = 1,
    GPIO_PIN_RA2 = 2,
    GPIO_PIN_RA3 = 3,
    GPIO_PIN_RA4 = 4,
    GPIO_PIN_RA5 = 5,
    GPIO_PIN_RA6 = 6,
    GPIO_PIN_RA7 = 7,
    GPIO_PIN_RA9 = 9,
    GPIO_PIN_RA10 = 10,
    GPIO_PIN_RA14 = 14,
    GPIO_PIN_RA15 = 15,
    GPIO_PIN_RB0 = 16,
    GPIO_PIN_RB1 = 17,
    GPIO_PIN_RB2 = 18,
    GPIO_PIN_RB3 = 19,
    GPIO_PIN_RB4 = 20,
    GPIO_PIN_RB5 = 21,
    GPIO_PIN_RB6 = 22,
    GPIO_PIN_RB7 = 23,
    GPIO_PIN_RB8 = 24,
    GPIO_PIN_RB9 = 25,
    GPIO_PIN_RB10 = 26,
    GPIO_PIN_RB11 = 27,
    GPIO_PIN_RB12 = 28,
    GPIO_PIN_RB13 = 29,
    GPIO_PIN_RB14 = 30,
    GPIO_PIN_RB15 = 31,
    GPIO_PIN_RC1 = 33,
    GPIO_PIN_RC2 = 34,
    GPIO_PIN_RC3 = 35,
    GPIO_PIN_RC4 = 36,
    GPIO_PIN_RC12 = 44,
    GPIO_PIN_RC13 = 45,
    GPIO_PIN_RC14 = 46,
    GPIO_PIN_RC15 = 47,
    GPIO_PIN_RD0 = 48,
    GPIO_PIN_RD1 = 49,
    GPIO_PIN_RD2 = 50,
    GPIO_PIN_RD3 = 51,
    GPIO_PIN_RD4 = 52,
    GPIO_PIN_RD5 = 53,
    GPIO_PIN_RD6 = 54,
    GPIO_PIN_RD7 = 55,
    GPIO_PIN_RD9 = 57,
    GPIO_PIN_RD10 = 58,
    GPIO_PIN_RD11 = 59,
    GPIO_PIN_RD12 = 60,
    GPIO_PIN_RD13 = 61,
    GPIO_PIN_RD14 = 62,
    GPIO_PIN_RD15 = 63,
    GPIO_PIN_RE0 = 64,
    GPIO_PIN_RE1 = 65,
    GPIO_PIN_RE2 = 66,
    GPIO_PIN_RE3 = 67,
    GPIO_PIN_RE4 = 68,
    GPIO_PIN_RE5 = 69,
    GPIO_PIN_RE6 = 70,
    GPIO_PIN_RE7 = 71,
    GPIO_PIN_RE8 = 72,
    GPIO_PIN_RE9 = 73,
    GPIO_PIN_RF0 = 80,
    GPIO_PIN_RF1 = 81,
    GPIO_PIN_RF2 = 82,
    GPIO_PIN_RF3 = 83,
    GPIO_PIN_RF4 = 84,
    GPIO_PIN_RF5 = 85,
    GPIO_PIN_RF8 = 88,
    GPIO_PIN_RF12 = 92,
    GPIO_PIN_RF13 = 93,
    GPIO_PIN_RG0 = 96,
    GPIO_PIN_RG1 = 97,
    GPIO_PIN_RG6 = 102,
    GPIO_PIN_RG7 = 103,
    GPIO_PIN_RG8 = 104,
    GPIO_PIN_RG9 = 105,
    GPIO_PIN_RG12 = 108,
    GPIO_PIN_RG13 = 109,
    GPIO_PIN_RG14 = 110,
    GPIO_PIN_RG15 = 111,
    GPIO_PIN_RH0 = 112,
    GPIO_PIN_RH1 = 113,
    GPIO_PIN_RH2 = 114,
    GPIO_PIN_RH3 = 115,
    GPIO_PIN_RH4 = 116,
    GPIO_PIN_RH5 = 117,
    GPIO_PIN_RH6 = 118,
    GPIO_PIN_RH7 = 119,
    GPIO_PIN_RH8 = 120,
    GPIO_PIN_RH9 = 121,
    GPIO_PIN_RH10 = 122,
    GPIO_PIN_RH11 = 123,
    GPIO_PIN_RH12 = 124,
    GPIO_PIN_RH13 = 125,
    GPIO_PIN_RH14 = 126,
    GPIO_PIN_RH15 = 127,
    GPIO_PIN_RJ0 = 128,
    GPIO_PIN_RJ1 = 129,
    GPIO_PIN_RJ2 = 130,
    GPIO_PIN_RJ3 = 131,
    GPIO_PIN_RJ4 = 132,
    GPIO_PIN_RJ5 = 133,
    GPIO_PIN_RJ6 = 134,
    GPIO_PIN_RJ7 = 135,
    GPIO_PIN_RJ8 = 136,
    GPIO_PIN_RJ9 = 137,
    GPIO_PIN_RJ10 = 138,
    GPIO_PIN_RJ11 = 139,
    GPIO_PIN_RJ12 = 140,
    GPIO_PIN_RJ13 = 141,
    GPIO_PIN_RJ14 = 142,
    GPIO_PIN_RJ15 = 143,
    GPIO_PIN_RK0 = 144,
    GPIO_PIN_RK1 = 145,
    GPIO_PIN_RK2 = 146,
    GPIO_PIN_RK3 = 147,
    GPIO_PIN_RK4 = 148,
    GPIO_PIN_RK5 = 149,
    GPIO_PIN_RK6 = 150,
    GPIO_PIN_RK7 = 151,

    /* This element should not be used in any of the GPIO APIs.
       It will be used by other modules or application to denote that none of the GPIO Pin is used */
    GPIO_PIN_NONE = -1

} GPIO_PIN;

typedef  void (*GPIO_PIN_CALLBACK) ( GPIO_PIN pin, uintptr_t context);

void GPIO_Initialize(void);

// *****************************************************************************
// *****************************************************************************
// Section: GPIO Functions which operates on multiple pins of a port
// *****************************************************************************
// *****************************************************************************

uint32_t GPIO_PortRead(GPIO_PORT port);

void GPIO_PortWrite(GPIO_PORT port, uint32_t mask, uint32_t value);

uint32_t GPIO_PortLatchRead ( GPIO_PORT port );

void GPIO_PortSet(GPIO_PORT port, uint32_t mask);

void GPIO_PortClear(GPIO_PORT port, uint32_t mask);

void GPIO_PortToggle(GPIO_PORT port, uint32_t mask);

void GPIO_PortInputEnable(GPIO_PORT port, uint32_t mask);

void GPIO_PortOutputEnable(GPIO_PORT port, uint32_t mask);

void GPIO_PortInterruptEnable(GPIO_PORT port, uint32_t mask);

void GPIO_PortInterruptDisable(GPIO_PORT port, uint32_t mask);

// *****************************************************************************
// *****************************************************************************
// Section: Local Data types and Prototypes
// *****************************************************************************
// *****************************************************************************

typedef struct {

    /* target pin */
    GPIO_PIN                 pin;

    /* Callback for event on target pin*/
    GPIO_PIN_CALLBACK        callback;

    /* Callback Context */
    uintptr_t               context;

} GPIO_PIN_CALLBACK_OBJ;

// *****************************************************************************
// *****************************************************************************
// Section: GPIO Functions which operates on one pin at a time
// *****************************************************************************
// *****************************************************************************

static inline void GPIO_PinWrite(GPIO_PIN pin, bool value)
{
    GPIO_PortWrite((GPIO_PORT)(pin>>4), (uint32_t)(0x1) << (pin & 0xF), (uint32_t)(value) << (pin & 0xF));
}

static inline bool GPIO_PinRead(GPIO_PIN pin)
{
    return (bool)(((GPIO_PortRead((GPIO_PORT)(pin>>4))) >> (pin & 0xF)) & 0x1);
}

static inline bool GPIO_PinLatchRead(GPIO_PIN pin)
{
    return (bool)((GPIO_PortLatchRead((GPIO_PORT)(pin>>4)) >> (pin & 0xF)) & 0x1);
}

static inline void GPIO_PinToggle(GPIO_PIN pin)
{
    GPIO_PortToggle((GPIO_PORT)(pin>>4), 0x1 << (pin & 0xF));
}

static inline void GPIO_PinSet(GPIO_PIN pin)
{
    GPIO_PortSet((GPIO_PORT)(pin>>4), 0x1 << (pin & 0xF));
}

static inline void GPIO_PinClear(GPIO_PIN pin)
{
    GPIO_PortClear((GPIO_PORT)(pin>>4), 0x1 << (pin & 0xF));
}

static inline void GPIO_PinInputEnable(GPIO_PIN pin)
{
    GPIO_PortInputEnable((GPIO_PORT)(pin>>4), 0x1 << (pin & 0xF));
}

static inline void GPIO_PinOutputEnable(GPIO_PIN pin)
{
    GPIO_PortOutputEnable((GPIO_PORT)(pin>>4), 0x1 << (pin & 0xF));
}

static inline void GPIO_PinInterruptEnable(GPIO_PIN pin)
{
    GPIO_PortInterruptEnable((GPIO_PORT)(pin>>4), 0x1 << (pin & 0xF));
}

static inline void GPIO_PinInterruptDisable(GPIO_PIN pin)
{
    GPIO_PortInterruptDisable((GPIO_PORT)(pin>>4), 0x1 << (pin & 0xF));
}

bool GPIO_PinInterruptCallbackRegister(
    GPIO_PIN pin,
    const   GPIO_PIN_CALLBACK callBack,
    uintptr_t context
);

// DOM-IGNORE-BEGIN
#ifdef __cplusplus  // Provide C++ Compatibility

    }

#endif
// DOM-IGNORE-END
#endif // PLIB_GPIO_H
