/******************************************************************************
  Filename:       ustv_filtres.c
  Revised:        $Date: 2012-11-19 18:52:21 $
  Revision:       $Revision: 00000 $

  Description:   Filters for embeded signal filtering

  Copyright 2012 Universit du Sud Toulon Var. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer .You may not use this
  Software unless you agree to abide by the terms of the License. Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED ?AS IS? WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact DPT GEII at www.univ-tln.fr.

 *****************************************************************************/

/*******************************************************************************
 * INCLUDES
 ******************************************************************************/
#include "ustv_filtres.h"
#include <math.h>
#include "Define.h"



#define  PI ((float)3.1415926535)


#ifdef USE_FILTER
unsigned int nbSerieDone_X=1;
unsigned int nbSerieDone_Y=1;
unsigned int nbSerieDone_Z=1;

/*******************************************************************************
 * FUNCTIONS
 ******************************************************************************/
#ifdef USE_Q16
/*******************************************************************************
 * @fn      FiltreOrdre1TpsReel
 *
 * @brief   Fonction executant le filtrage d'un signal. (a executer a chaque
 *          echantillon). Filtre d'ordre 1.
 *
 * @param   filtre - struct FiltreOrdre1 - variable/objet filtre qui contiendra
 *          les infos/parametres du filtre.
 *          e_0 - _Q16 - Echantillon a filtrer
 *
 * @return  _Q16 - echantillon filtr
 *
 ******************************************************************************/
_Q16 FiltreOrdre1TpsReel(FiltreOrdre1 * filtre, _Q16 e_0)
{
    switch(filtre->initOutputMode)
    {
        case 0 : //Filtre initialis
            break;
        case 1: //Initialisation Filtre passe-bas
            filtre->initOutputMode = 0;
            filtre->s_1 = e_0;
            filtre->e_1 = e_0;
            break;
        case 2: //Initialisation Filtre passe-haut
            filtre->initOutputMode = 0;
            filtre->s_1 = 0;
            filtre->e_1 = e_0;
            break;
        default :
            filtre->initOutputMode = 0;
            break;
    }

    //On calcule la sortie du filtre
    //filtre->s_0 = _Q16mac(filtre->_d_0 , _Q16mac(e_0 ,filtre->n_0,_Q16mac(filtre->e_1 , filtre->n_1,_Q16neg(_Q16mac(filtre->s_1 , filtre->d_1,NULLACCU)))),NULLACCU);
    filtre->s_0 = _Q16div(_Q16mac(e_0 ,filtre->n_0,_Q16mac(filtre->e_1 , filtre->n_1,_Q16neg(_Q16mac(filtre->s_1 , filtre->d_1,NULLACCU)))),filtre->d_0);

    //On ractualise les variables internes pour le cycle suivant
    filtre->s_1 = filtre->s_0;
    filtre->e_1 = e_0;

    return filtre->s_0;
}
#endif

double FiltreOrdre1TpsReeldbl(FiltreOrdre1dbl * filtre, double e_0)
{
    switch(filtre->initOutputMode)
    {
        case 0 : //Filtre initialis
            break;
        case 1: //Initialisation Filtre passe-bas
            filtre->initOutputMode = 0;
            filtre->s_1 = e_0;
            filtre->e_1 = e_0;
            break;
        case 2: //Initialisation Filtre passe-haut
            filtre->initOutputMode = 0;
            filtre->s_1 = 0;
            filtre->e_1 = e_0;
            break;
        default :
            filtre->initOutputMode = 0;
            break;
    }

    //On calcule la sortie du filtre
    filtre->s_0 = 1.0 / filtre->d_0 * (e_0 *filtre->n_0 + filtre->e_1 * filtre->n_1 - filtre->s_1 * filtre->d_1);

    //On ractualise les variables internes pour le cycle suivant
    filtre->s_1 = filtre->s_0;
    filtre->e_1 = e_0;

    return filtre->s_0;
}

#ifdef USE_Q16
_Q16 FiltreOrdre1TpsReelScaled(FiltreOrdre1 * filtre, _Q16 e_0, _Q16 scale)
{
    //Dans le cas o on vient saturer les termes de la somme n0*e0 + n1*e1 + d1* s1, il faut scaler les entres
    //C'est en particulier valable si n0, n1 ou d1 sont trs grands !

    e_0 = _Q16div(e_0, scale);
    //On calcule la sortie du filtre
    //filtre->s_0 = _Q16mac(filtre->_d_0 , _Q16mac(e_0 ,filtre->n_0,_Q16mac(filtre->e_1 , filtre->n_1,_Q16neg(_Q16mac(filtre->s_1 , filtre->d_1,NULLACCU)))),NULLACCU);
    filtre->s_0 = _Q16div(_Q16mac(e_0 ,filtre->n_0,_Q16mac(filtre->e_1 , filtre->n_1,_Q16neg(_Q16mac(filtre->s_1 , filtre->d_1,NULLACCU)))),filtre->d_0);

    //On ractualise les variables internes pour le cycle suivant
    filtre->s_1 = filtre->s_0;
    filtre->e_1 = e_0;

    return _Q16mac(filtre->s_0, scale, NULLACCU);
}

/*******************************************************************************
 * @fn      FiltreOrdre2TpsReel
 *
 * @brief   Fonction executant le filtrage d'un signal. (a executer a chaque
 *          echantillon). Filtre d'ordre 2.
 *
 * @param   filtre - struct FiltreOrdre2 - variable/objet filtre qui contiendra
 *          les infos/parametres du filtre.
 *          e_0 - double - Echantillon a filtrer
 *
 * @return  double - echantillon filtr
 *
 ******************************************************************************/
_Q16 FiltreOrdre2TpsReel(FiltreOrdre2 * filtre, _Q16 e_0)
{
    //On calcule la sortie du filtre
//    filtre->s_0 = _Q16mac(filtre->_d_0 ,_Q16mac(e_0 ,filtre->n_0,_Q16mac(filtre->e_1 ,filtre->n_1,_Q16mac(filtre->e_2 ,filtre->n_2,_Q16neg(_Q16mac(filtre->s_1 ,filtre->d_1,_Q16neg(_Q16mac(filtre->s_2 ,filtre->d_2,NULLACCU))))))),NULLACCU);
    filtre->s_0 = _Q16div(_Q16mac(e_0 ,filtre->n_0,_Q16mac(filtre->e_1 ,filtre->n_1,_Q16mac(filtre->e_2 ,filtre->n_2,_Q16neg(_Q16mac(filtre->s_1 ,filtre->d_1,_Q16mac(filtre->s_2 ,filtre->d_2,NULLACCU)))))),filtre->_d_0);
    //On ractualise les variables internes pour le cycle suivant
    filtre->s_2 = filtre->s_1;
    filtre->s_1 = filtre->s_0;
    filtre->e_2 = filtre->e_1;
    filtre->e_1 = e_0;

    return filtre->s_0;
}
#endif
/*******************************************************************************
 * @fn      FiltreOrdre2TpsReel
 *
 * @brief   Fonction executant le filtrage d'un signal. (a executer a chaque
 *          echantillon). Filtre d'ordre 2.
 *
 * @param   filtre - struct FiltreOrdre2 - variable/objet filtre qui contiendra
 *          les infos/parametres du filtre.
 *          e_0 - double - Echantillon a filtrer
 *
 * @return  double - echantillon filtr
 *
 ******************************************************************************/
double FiltreOrdre2TpsReeldbl(FiltreOrdre2dbl * filtre, double e_0)
{

     //On calcule la sortie du filtre
    filtre->s_0 = 1.0 / filtre->d_0 * (e_0 *filtre->n_0 + filtre->e_1 * filtre->n_1 + filtre->e_2 * filtre->n_2 - filtre->s_1 * filtre->d_1 - filtre->s_2 * filtre->d_2);

    //On ractualise les variables internes pour le cycle suivant
    filtre->s_2 = filtre->s_1;
    filtre->s_1 = filtre->s_0;
    filtre->e_2 = filtre->e_1;
    filtre->e_1 = e_0;

    return filtre->s_0;
}
/*******************************************************************************
 * @fn      FiltreOrdre1SetOutput
 *
 * @brief   Fonction actualisant la sortie du filtre d'ordre 1.
 *
 * @param   filtre - struct FiltreOrdre1 - variable/objet filtre qui contiendra
 *          les infos/parametres du filtre.
 *          value - double - valeure a mettre en sortie.
 *
 * @return  None.
 *
 ******************************************************************************/
#ifdef USE_Q16
void FiltreOrdre1SetOutput(FiltreOrdre1 * filtre, double value)
{
    //Init des sorties
    filtre->s_0=_Q16ftoi(value);
    filtre->s_1=_Q16ftoi(value);

    //On init aussi les entre
    filtre->e_1=_Q16ftoi(value);
}
#endif

void FiltreOrdre1SetOutputdbl(FiltreOrdre1dbl * filtre, double value)
{
    //Init des sorties
    filtre->s_0=value;
    filtre->s_1=value;

    //On init aussi les entres
    filtre->e_1=value;
}

#ifdef USE_Q16
void FiltreOrdre2SetOutput(FiltreOrdre2 * filtre, double value)
{
    //Init des sorties
    filtre->s_0 = _Q16ftoi(value);
    filtre->s_1 = _Q16ftoi(value);
    filtre->s_2 = _Q16ftoi(value);
}

void FiltrePasseHautButterworthOrdre1SetFrequence(FiltreOrdre1 * filtre, double freqEch, double freqCoupure)
{
    //TODO: grer le cas des frequences nulles dans le filtres -> division par 0
    filtre->w0 = _Q16ftoi(freqCoupure); //6.28 * freqCoupure;
    filtre->ts = _Q16ftoi(1/freqEch);
    filtre->c =  _Q16div(_Q16_CONST_1,_Q16tan(_Q16div(_Q16mac(filtre->w0,filtre->ts,NULLACCU),_Q16_CONST_2)));

    filtre->n_0 = filtre->c;
    filtre->n_1 = _Q16neg(filtre->c);

    filtre->_d_0 = _Q16div(_Q16_CONST_1,(filtre->c + _Q16_CONST_1));
    filtre->d_0 = filtre->c + _Q16_CONST_1;
    filtre->d_1 = _Q16neg(filtre->c) + _Q16_CONST_1;
}

void FiltrePasseBasButterworthOrdre1SetFrequence(FiltreOrdre1 * filtre, double freqEch, double freqCoupure)
{
    //TODO: grer le cas des frequences nulles dans le filtres -> division par 0
    filtre->w0 = _Q16ftoi(freqCoupure); //6.28 * freqCoupure;
    filtre->ts = _Q16ftoi(1/freqEch);
    filtre->c =  _Q16div(_Q16_CONST_1,_Q16tan(_Q16div(_Q16mac(filtre->w0,filtre->ts,NULLACCU),_Q16_CONST_2)));

    filtre->n_0 = _Q16ftoi(1);
    filtre->n_1 = _Q16ftoi(1);

    filtre->_d_0 = _Q16div(_Q16_CONST_1,(filtre->c + _Q16_CONST_1));
    filtre->d_0 = filtre->c + _Q16_CONST_1;
    filtre->d_1 = _Q16neg(filtre->c) + _Q16_CONST_1;
}
#endif

void FiltrePasseBasButterworthOrdre1SetFrequencedbl(FiltreOrdre1dbl * filtre, double freqEch, double freqCoupure)
{
    filtre->w0 = freqCoupure; //6.28 * freqCoupure;
    filtre->ts = 1/freqEch;
    filtre->c = 1/tan(filtre->w0*filtre->ts/2);

    filtre->n_0 = 1;
    filtre->n_1 = 1;

    filtre->d_0 = filtre->c + 1;
    filtre->d_1 = -filtre->c + 1;
}

#ifdef USE_Q16
static FiltreOrdre2 debugFilter;
void FiltrePasseHautButterworthOrdre2SetFrequence(FiltreOrdre2 * filtre, double freqEch, double freqCoupure)
{

    filtre->w0 = _Q16ftoi(freqCoupure); //6.28 * freqCoupure;
    filtre->ts = _Q16ftoi(1.0/freqEch);
    filtre->c = _Q16div(_Q16_CONST_1,_Q16tan(_Q16div(_Q16mac(filtre->w0,filtre->ts,NULLACCU),_Q16_CONST_2)));
    filtre->Q = _Q16ftoi(5);                                  //??

    filtre->n_0 = _Q16mac(filtre->c ,filtre->c,NULLACCU);
    filtre->n_1 = _Q16mac(_Q16neg(_Q16_CONST_2) ,_Q16mac(filtre->c ,filtre->c,NULLACCU),NULLACCU);
    filtre->n_2 = _Q16mac(filtre->c ,filtre->c,NULLACCU);

    filtre->_d_0 = _Q16div(_Q16_CONST_1,_Q16mac(filtre->c ,filtre->c,_Q16mac(_Q16ftoi(1.41421) ,filtre->c,_Q16_CONST_1)));
    filtre->d_1 = _Q16mac(_Q16neg(_Q16_CONST_2) ,(_Q16mac(filtre->c ,filtre->c,_Q16neg(_Q16_CONST_1))),NULLACCU);
    filtre->d_2 = _Q16mac(filtre->c ,filtre->c,_Q16neg(_Q16mac(_Q16ftoi(1.41421) ,filtre->c,_Q16_CONST_1)));
    debugFilter=*filtre;
}

void FiltrePasseBasButterworthOrdre2SetFrequence(FiltreOrdre2 * filtre, double freqEch, double freqCoupure)
{
    filtre->w0 = _Q16ftoi(freqCoupure); //6.28 * freqCoupure;
    filtre->ts = _Q16ftoi(1.0/freqEch);
    filtre->c = _Q16div(_Q16_CONST_1,_Q16tan(_Q16div(_Q16mac(filtre->w0,filtre->ts,NULLACCU),_Q16_CONST_2)));
    filtre->Q = _Q16ftoi(5.0);                                  //??

    filtre->n_0 = _Q16_CONST_1;
    filtre->n_1 = _Q16_CONST_2;
    filtre->n_2 = _Q16_CONST_1;

    filtre->_d_0 = _Q16div(_Q16_CONST_1,_Q16mac(filtre->c,filtre->c,_Q16mac(_Q16ftoi(1.41421) ,filtre->c,_Q16_CONST_1)));
    filtre->d_1 = _Q16mac(_Q16neg(_Q16_CONST_2) ,(_Q16mac(filtre->c ,filtre->c,NULLACCU) - _Q16_CONST_1),NULLACCU);
    filtre->d_2 = _Q16mac(filtre->c,filtre->c,_Q16neg(_Q16mac(_Q16ftoi(1.41421) ,filtre->c,_Q16_CONST_1)));

}
#endif

/*******************************************************************************
 * @fn      FiltrePasseHautButterworthOrdre1Init
 *
 * @brief   Fonction permettant l'initialisation du filtre passe haut d'ordre 1.
 *          Celle-ci calcule les parametres du filtre a partir des parametre:
 *          -frequence d'echantillonage
 *          -frequence de coupure
 *
 * @param   filtre - struct FiltreOrdre1 - variable/objet filtre qui contiendra
 *          les infos/parametres du filtre.
 *          fEch - double - frequence d'echantillonage
 *          f0 - double - Frequence de coupure du filtre.
 *
 * @return  None.
 *
 ******************************************************************************/
#ifdef USE_Q16
void FiltrePasseHautButterworthOrdre1Init(FiltreOrdre1 * filtre, double freqEch, double freqCoupure)
{
    FiltrePasseHautButterworthOrdre1SetFrequence(filtre, freqEch, freqCoupure);
    FiltrePasseHautButterworthOrdre1OutputInit(filtre);
}

void FiltrePasseHautButterworthOrdre1OutputInit(FiltreOrdre1 * filtre)
{
    filtre->initOutputMode = 2; //passe-haut
}
#endif

void FiltrePasseHautButterworthOrdre1Initdbl(FiltreOrdre1dbl * filtre, double freqEch, double freqCoupure)
{
    filtre->w0 = freqCoupure; //6.28 * freqCoupure;
    filtre->ts = 1/freqEch;
    filtre->c = 1/tan(filtre->w0*filtre->ts/2);

    filtre->n_0 = filtre->c;
    filtre->n_1 = -filtre->c;

    filtre->d_0 = filtre->c + 1;
    filtre->d_1 = -filtre->c + 1;

    //Init des sorties
    filtre->s_0 = 0;
    filtre->s_1 = 0;

    //Init des entrees
 //   filtre->e_0 = 0;
    filtre->e_1 = 0;
}

/*******************************************************************************
 * @fn      FiltrePasseHautButterworthOrdre2Init
 *
 * @brief   Fonction permettant l'initialisation du filtre passe haut d'ordre 2.
 *          Celle-ci calcule les parametres du filtre a partir des parametre:
 *          -frequence d'echantillonage
 *          -frequence de coupure
 *
 * @param   filtre - struct FiltreOrdre2 - variable/objet filtre qui contiendra
 *          les infos/parametres du filtre.
 *          fEch - double - frequence d'echantillonage
 *          f0 - double - Frequence de coupure du filtre.
 *
 * @return  None.
 *
 ******************************************************************************/
#ifdef USE_Q16
void FiltrePasseHautButterworthOrdre2Init(FiltreOrdre2 * filtre, double freqEch, double freqCoupure)
{
    FiltrePasseHautButterworthOrdre2SetFrequence(filtre, freqEch, freqCoupure);
    FiltreOrdre2SetOutput(filtre, 0);
}
#endif

void FiltrePasseHautButterworthOrdre2Initdbl(FiltreOrdre2dbl * filtre, double freqEch, double freqCoupure)
{
    filtre->w0 = freqCoupure; //6.28 * freqCoupure;                  //Pulsation du filtre
    filtre->ts = 1/freqEch;                           //Sample time
    filtre->c = 1/tan(filtre->w0*filtre->ts/2.0);     //Warping equation
    filtre->Q = 5.0;                                  //??

    filtre->n_0 = filtre->c * filtre->c;
    filtre->n_1 = -2.0 * filtre->c * filtre->c;
    filtre->n_2 = filtre->c * filtre->c;

    filtre->d_0 = filtre->c * filtre->c + 1.41421 * filtre->c + 1;
    filtre->d_1 = -2.0 * (filtre->c * filtre->c - 1);
    filtre->d_2 = filtre->c * filtre->c - 1.41421 * filtre->c + 1;

    //Init des sorties
    filtre->s_0 = 0;
    filtre->s_1 = 0;
    filtre->s_2 = 0;

    //Init des entrees
    filtre->e_1 = 0;
    filtre->e_2 = 0;
}
/*******************************************************************************
 * @fn      FiltrePasseBasButterworthOrdre1Init
 *
 * @brief   Fonction permettant l'initialisation du filtre passe haut d'ordre 1.
 *          Celle-ci calcule les parametres du filtre a partir des parametre:
 *          -frequence d'echantillonage
 *          -frequence de coupure
 *
 * @param   filtre - struct FiltreOrdre1 - variable/objet filtre qui contiendra
 *          les infos/parametres du filtre.
 *          fEch - double - frequence d'echantillonage
 *          f0 - double - Frequence de coupure du filtre.
 *
 * @return  None.
 *
 ******************************************************************************/
#ifdef USE_Q16
void FiltrePasseBasButterworthOrdre1Init(FiltreOrdre1 * filtre, double freqEch, double freqCoupure)
{
    FiltrePasseBasButterworthOrdre1SetFrequence(filtre, freqEch, freqCoupure);
    FiltrePasseBasButterworthOrdre1OutputInit(filtre);
}

void FiltrePasseBasButterworthOrdre1OutputInit(FiltreOrdre1 * filtre)
{
    filtre->initOutputMode = 1; //passe-bas
}
#endif

void FiltrePasseBasButterworthOrdre1Initdbl(FiltreOrdre1dbl * filtre, double freqEch, double freqCoupure)
{
    FiltrePasseBasButterworthOrdre1SetFrequencedbl(filtre, freqEch, freqCoupure);
    FiltrePasseBasButterworthOrdre1OutputInitdbl(filtre);
}

void FiltrePasseBasButterworthOrdre1OutputInitdbl(FiltreOrdre1dbl * filtre)
{
    filtre->initOutputMode = 1; //passe-bas
}

void FiltrePasseBandeButterworthOrdre4Initdbl(FiltreOrdre4dbl * filtre)
{
    filtre->e_1 = 0;
    filtre->e_2 = 0;
    filtre->e_3 = 0;
    filtre->e_4 = 0;
 
    filtre->s_0 = 0;
    filtre->s_1 = 0;
    filtre->s_2 = 0;
    filtre->s_3 = 0;
    filtre->s_4 = 0;
}

//void FiltrePasseBandeButterworthOrdre8Initdbl(FiltreOrdre8dbl * filtre)
//{
//    filtre->e_1 = 0;
//    filtre->e_2 = 0;
//    filtre->e_3 = 0;
//    filtre->e_4 = 0;
//    filtre->e_5 = 0;
//    filtre->e_6 = 0;
//    filtre->e_7 = 0;
//    filtre->e_8 = 0;
//
//    filtre->s_0 = 0;
//    filtre->s_1 = 0;
//    filtre->s_2 = 0;
//    filtre->s_3 = 0;
//    filtre->s_4 = 0;
//    filtre->s_5 = 0;
//    filtre->s_6 = 0;
//    filtre->s_7 = 0;
//    filtre->s_8 = 0;
//}
/*******************************************************************************
 * @fn      FiltrePasseBasButterworthOrdre1Init
 *
 * @brief   Fonction permettant l'initialisation du filtre passe haut d'ordre 2.
 *          Celle-ci calcule les parametres du filtre a partir des parametre:
 *          -frequence d'echantillonage
 *          -frequence de coupure
 *
 * @param   filtre - struct FiltreOrdre2 - variable/objet filtre qui contiendra
 *          les infos/parametres du filtre.
 *          fEch - double - frequence d'echantillonage
 *          f0 - double - Frequence de coupure du filtre.
 *
 * @return  None.
 *
 ******************************************************************************/
#ifdef USE_Q16
void FiltrePasseBasButterworthOrdre2Init(FiltreOrdre2 * filtre, double freqEch, double freqCoupure)
{
    FiltrePasseBasButterworthOrdre2SetFrequence(filtre, freqEch, freqCoupure);
    FiltreOrdre2SetOutput(filtre, 0);
}
#endif
void FiltrePasseBasButterworthOrdre2Initdbl(FiltreOrdre2dbl * filtre, double freqEch, double freqCoupure)
{

    filtre->w0 = freqCoupure; //6.28 * freqCoupure;                           //Pulsation du filtre
    filtre->ts = 1/freqEch;                           //Sample time
    filtre->c = 1/tan(filtre->w0*filtre->ts/2.0);     //Warping equation
    filtre->Q = 5.0;                                  //??

    filtre->n_0 = 1.0;
    filtre->n_1 = 2.0;
    filtre->n_2 = 1.0;

    filtre->d_0 = filtre->c * filtre->c + 1.41421 * filtre->c + 1;
    filtre->d_1 = -2.0 * (filtre->c * filtre->c - 1);
    filtre->d_2 = filtre->c * filtre->c - 1.41421 * filtre->c + 1;

    //Init des sorties
    filtre->s_0 = 0;
    filtre->s_1 = 0;
    filtre->s_2 = 0;

    //Init des entrees
    filtre->e_1 = 0;
    filtre->e_2 = 0;
}


void FiltreNotchOrdre2Initdbl(FiltreOrdre2dbl * filtre, double freqEch, double freqNotch)
{

    filtre->w0 = freqNotch * 6.28; //6.28 * freqCoupure;                           //Pulsation du filtre
    filtre->ts = 1/freqEch;                           //Sample time
    filtre->c = 1/tan(filtre->w0*filtre->ts/2.0);     //Warping equation
    filtre->Q = 1.0;                                  //??

    filtre->n_0 = filtre->Q * (filtre->c * filtre->c + 1);
    filtre->n_1 = -2 * filtre->Q * (filtre->c * filtre->c - 1);
    filtre->n_2 = filtre->Q * (filtre->c * filtre->c + 1);

    filtre->d_0 = filtre->c * filtre->c * filtre->Q + filtre->c + filtre->Q;
    filtre->d_1 = -2.0 * filtre->Q * (filtre->c * filtre->c - 1);
    filtre->d_2 = filtre->c * filtre->c * filtre->Q - filtre->c + filtre->Q;

    //Init des sorties
    filtre->s_0 = 0;
    filtre->s_1 = 0;
    filtre->s_2 = 0;

    //Init des entrees
    filtre->e_1 = 0;
    filtre->e_2 = 0;
}
double Filtre_Specific_BP4_Tchebytchev_8Hz_20Hz_Ech125Hz_TpsReel(FiltreOrdre4dbl * filtre, double e_0)
{
    //On calcule la sortie du filtre
    e_0 /= 8.518480096e+00;
    filtre->s_0 = (filtre->e_4 + e_0) + - 2 * filtre->e_2
             + (-0.3941329430 * filtre->s_4) + (1.3273184073 * filtre->s_3)
             + (-2.3170692186 * filtre->s_2) + (2.2682838861 * filtre->s_1);

    //On ractualise les variables internes pour le cycle suivant
    filtre->s_4 = filtre->s_3;
    filtre->s_3 = filtre->s_2;
    filtre->s_2 = filtre->s_1;
    filtre->s_1 = filtre->s_0;
    filtre->e_4 = filtre->e_3;
    filtre->e_3 = filtre->e_2;
    filtre->e_2 = filtre->e_1;
    filtre->e_1 = e_0;

    return filtre->s_0;
}

double Filtre_Specific_BP4_Tchebytchev_8Hz_20Hz_Ech250Hz_TpsReel(FiltreOrdre4dbl * filtre, double e_0)
{
    //On calcule la sortie du filtre

    e_0 /= 3.607295967e+01;
    filtre->s_0 = (filtre->e_4 + e_0) + - 2 * filtre->e_2
             + (-0.6538538826 * filtre->s_4) + (2.7058012676 * filtre->s_3)
             + (-4.4232336697 * filtre->s_2) + (3.3631390648 * filtre->s_1);


    //On ractualise les variables internes pour le cycle suivant
    filtre->s_4 = filtre->s_3;
    filtre->s_3 = filtre->s_2;
    filtre->s_2 = filtre->s_1;
    filtre->s_1 = filtre->s_0;
    filtre->e_4 = filtre->e_3;
    filtre->e_3 = filtre->e_2;
    filtre->e_2 = filtre->e_1;
    filtre->e_1 = e_0;

    return filtre->s_0;
}

//double Filtre_Specific_BP8_Tchebytchev_8Hz_20Hz_Ech125Hz_TpsReel(FiltreOrdre8dbl filtre, double e_0)
//{
//    //On calcule la sortie du filtre
//    e_0 /= 4.540077773e+02;
//    filtre.s_0 = (filtre.e_8 + e_0) - 4 * (filtre.e_6 + filtre.e_2) + 6 * filtre.e_4
//             + (-0.4871671369 * filtre.s_8) + (3.2323659678 * filtre.s_7)
//             + (-10.2846142690 * filtre.s_6) + (20.1965354020 * filtre.s_5)
//             + (-26.6673752670 * filtre.s_4) + (24.2061664630 * filtre.s_3)
//             + (-14.7766198110 * filtre.s_2) + (5.5639349329 * filtre.s_1);
//
//    //On ractualise les variables internes pour le cycle suivant
//    filtre.s_8 = filtre.s_7;
//    filtre.s_7 = filtre.s_6;
//    filtre.s_6 = filtre.s_5;
//    filtre.s_5 = filtre.s_4;
//    filtre.s_4 = filtre.s_3;
//    filtre.s_3 = filtre.s_2;
//    filtre.s_2 = filtre.s_1;
//    filtre.s_1 = filtre.s_0;
//    filtre.e_8 = filtre.e_7;
//    filtre.e_7 = filtre.e_6;
//    filtre.e_6 = filtre.e_5;
//    filtre.e_5 = filtre.e_4;
//    filtre.e_4 = filtre.e_3;
//    filtre.e_3 = filtre.e_2;
//    filtre.e_2 = filtre.e_1;
//    filtre.e_1 = e_0;
//
//    return filtre.s_0;
//}

#ifdef USE_Q16
/*******************************************************************************
 * @fn      IntegrateurInit
 *
 * @brief   Fonction permettant l'initialisation de l'integrateur
 *
 * @param   filtre - struct Integrateur - variable/objet de l'integrateur.
 *          s_inin - double - ??
 *
 * @return  -.
 *
 ******************************************************************************/
void _Q16IntegrateurInit(_Q16Integrateur * filtre, double freqEch)
{
    //_d_0  = 1/(2.0*fech)
    filtre->_d_0 = _Q16div(_Q16_CONST_1, _Q16ftoi(freqEch*2));
    //Init des sorties
    filtre->s_0 = 0;
    //Init des entrees
    filtre->e_1 = 0;
}

/*******************************************************************************
 * @fn      Integrate
 *
 * @brief   Fonction permettant l'integration d'un signal.
 *
 * @param   filtre - struct Integrateur - variable/objet de l'integrateur.
 *          e_0 - double - echantillon a integrer.
 *
 * @return  valeur de l'integration de l'echantillon au precedant
 *          (signal integr).
 *
 ******************************************************************************/
_Q16 _Q16Integrate(_Q16Integrateur * filtre, _Q16 e_0)
{
    //On calcule la sortie du filtre
    filtre->s_0 = _Q16mac(_Q16mac(filtre->e_1, _Q16_CONST_1,  e_0), filtre->_d_0, filtre->s_0);

    //filtre->s_0 += (filtre->e_1 + e_0)/2.0 * (1.0 / (double)filtre->fech);
    //On ractualise les variables internes pour le cycle suivant
    filtre->e_1 = e_0;
    //On renvoie la valeur
    return filtre->s_0;
}

/*******************************************************************************
 * @fn      IntegrateurInit
 *
 * @brief   Fonction permettant l'initialisation de l'integrateur
 *
 * @param   filtre - struct Integrateur - variable/objet de l'integrateur.
 *          s_inin - double - ??
 *
 * @return  -.
 *
 ******************************************************************************/
void _Q16IntegrateurReset(_Q16Integrateur * filtre)
{
    //Init des sorties
    filtre->s_0 = 0;
    //Init des entrees
    filtre->e_1 = 0;
}
#endif

/*******************************************************************************
 * @fn      IntegrateurInit
 *
 * @brief   Fonction permettant l'initialisation de l'integrateur
 *
 * @param   filtre - struct Integrateur - variable/objet de l'integrateur.
 *          s_inin - double - ??
 *
 * @return  -.
 *
 ******************************************************************************/
void IntegrateurInit(Integrateur * filtre, double freqEch)
{
    //_d_0  = 1/(2.0*fech)
    filtre->fech = freqEch;
    filtre->_d_0 = 1.0/(freqEch*2.0);
    //Init des sorties
    filtre->s_0 = 0;
    //Init des entrees
    filtre->e_1 = 0;
}

/*******************************************************************************
 * @fn      Integrate
 *
 * @brief   Fonction permettant l'integration d'un signal.
 *
 * @param   filtre - struct Integrateur - variable/objet de l'integrateur.
 *          e_0 - double - echantillon a integrer.
 *
 * @return  valeur de l'integration de l'echantillon au precedant
 *          (signal integr).
 *
 ******************************************************************************/
double Integrate(Integrateur * filtre, double e_0)
{
    //On calcule la sortie du filtre
    //filtre->s_0 = _Q16mac(_Q16mac(filtre->e_1, _Q16_CONST_1,  e_0), filtre->_d_0, filtre->s_0);

    filtre->s_0 += (filtre->e_1 + e_0)/2.0 * (1.0 / (double)filtre->fech);
    //filtre->s_0 += (filtre->e_1 + e_0)*filtre->_d_0;
    //On ractualise les variables internes pour le cycle suivant
    filtre->e_1 = e_0;
    //On renvoie la valeur
    return filtre->s_0;
}

/*******************************************************************************
 * @fn      DerivateurInit
 *
 * @brief   Fonction permettant l'initialisation de l'integrateur
 *
 * @param   filtre - struct Integrateur - variable/objet de l'integrateur.
 *          s_inin - double - ??
 *
 * @return  -.
 *
 ******************************************************************************/
void DerivateurInit(Derivateur * filtre, double freqEch)
{
    //_d_0  = 1/(2.0*fech)
    filtre->fech = freqEch;
    //Init des entrees
    filtre->e_1 = 0;
}

/*******************************************************************************
 * @fn      Integrate
 *
 * @brief   Fonction permettant l'integration d'un signal.
 *
 * @param   filtre - struct Integrateur - variable/objet de l'integrateur.
 *          e_0 - double - echantillon a integrer.
 *
 * @return  valeur de l'integration de l'echantillon au precedant
 *          (signal integr).
 *
 ******************************************************************************/
double Derivate(Derivateur * filtre, double e_0)
{
    double output = (e_0 - filtre->e_1);//*filtre->fech;
    //On ractualise les variables internes pour le cycle suivant
    filtre->e_1 = e_0;
    //On renvoie la valeur
    return output;
}

/*******************************************************************************
 * @fn      FiltreMinMaxInitOrdre2
 *
 * @brief   Fonction permettant l'initialisation du filtre de detection
 *          d'enveloppe d'ordre 2.(detection du min et du max).
 *          Celle-ci initialise les parametres du filtre a partir des parametre:
 *          -frequence d'echantillonage
 *          -frequence de coupure
 *
 * @param   filtre - struct FiltreMinMaxOrdre2 - variable/objet filtre qui contiendra
 *          les infos/parametres du filtre.
 *          fEch - double - frequence d'echantillonage
 *          f0 - double - Frequence de coupure du filtre.
 *
 * @return  None.
 *
 ******************************************************************************/
#ifdef USE_Q16
void FiltreMinMaxOrdre1Init(FiltreMinMaxOrdre1 * filtre, double freqEch, double f0, long deadZoneDelay, double minTriggerValue)
{
    filtre->aMax = filtre->aMin = 0;
    filtre->flagCroissant = FALSE;
    filtre->flagDecroissant = FALSE;
    filtre->trigState = Waiting;
    filtre->deadZoneDelay = deadZoneDelay;
    filtre->amplitudeMin = _Q16ftoi(minTriggerValue);
    filtre->timeStampLastTriggerDown = (unsigned int)g_longTimeStamp;
    filtre->timeStampLastTriggerUp = (unsigned int)g_longTimeStamp;
}
#endif

void FiltreMinMaxOrdre1Initdbl(FiltreMinMaxOrdre1dbl * filtre, double freqEch, double f0)
{
    filtre->aMax = filtre->aMin = filtre->aMaxMoy = filtre->aMinMoy = 0;
    filtre->flagCroissant = FALSE;
    filtre->flagDecroissant = FALSE;
    filtre->trigState = Waiting;
}

#ifdef USE_Q16
void FiltreMinMaxOrdre2Init(FiltreMinMaxOrdre2 * filtre, double freqEch, double f0)
{
    //FiltrePasseBasButterworthOrdre2Init(&filtre->filtreMax, fEch, f0);
    //FiltrePasseBasButterworthOrdre2Init(&filtre->filtreMin, fEch, f0);
    filtre->aMax = filtre->aMin = filtre->aMaxMoy = filtre->aMinMoy = 0;
    filtre->flagCroissant = FALSE;
    filtre->flagDecroissant = FALSE;
    filtre->trigState = Waiting;
}
#endif

/*******************************************************************************
 * @fn      FiltreMinMaxOrdr1
 *
 * @brief   Fonction effectuant la detection d'enveloppe d'ordre 1.
 *          (detection du min et du max).
 *          Celle-ci actualise les parametres du filtre a partir de la valeure
 *          d'entre du filtre

 * @param   filtre - struct FiltreMinMaxOrdre1 - variable/objet filtre qui contiendra
 *          les infos/parametres du filtre.
 *          value - double - valeure d'entree du filtre (1 echantillon).
 *
 * @return  struct FiltreMinMaxOrdre1 - variable/objet filtre qui contiendra
 *          les infos/parametres du filtre actualis.
 *
 ******************************************************************************/
void FiltreMinMaxOrdre1Process(FiltreMinMaxOrdre1dbl * filtre, double value)
{
    FiltreMinMaxOrdre1CalculCroissantdbl(filtre,value,0);
    FiltreMinMaxOrdre1CalculDecroissantdbl(filtre,value,0);
    //return filtre;
}

#ifdef USE_Q16
unsigned char FiltreMinMaxOrdre1CalculCroissant(FiltreMinMaxOrdre1 * filtre, _Q16 input, long longTimeStamp)
{
    //Variable pour savoir si on est dans la dead zone ou pas
    unsigned char updateOutput = 0;
    if (longTimeStamp > filtre->deadZoneDelay + filtre->timeStampLastTriggerUp)
        updateOutput = 1;

    //On est hors de la dead zone
    if(updateOutput)
    {
        filtre->filtreMax->s_0 = FiltreOrdre1TpsReel(filtre->filtreMax, filtre->filtreMin->s_0);
    }
    //calcul de Vmax sur l'axe Z
    if (input >= filtre->filtreMax->s_0)          //ATTENTION COMPARAISON EN Q16
    {
        //VmaxMoyFiltree = Vmax = vit;
        FiltreOrdre1SetOutput(filtre->filtreMax, _itofQ16(input));      //Attention, la fonction attend un double..
        filtre->aMax = input;
        filtre->flagCroissant = TRUE;
    }
    else
    {
        if (updateOutput == TRUE)
        {
            if (filtre->flagCroissant == TRUE)
            {
                //On vient de passer le max
                filtre->flagCroissant = FALSE;
                filtre->aMax = filtre->filtreMax->s_0;
                //On teste si l'amplitude minimale a t respecte
                if (_Q16mac(filtre->aMax, _Q16_CONST_1, _Q16neg(filtre->filtreMin->s_0)) > filtre->amplitudeMin)
                {
                    if (filtre->trigStateStart == Waiting)
                    {
                        //On mmorise aussi la premire transition
                        filtre->trigStateStart = TriggeredHigh;
                        filtre->trigState = TriggeredHigh;
                        filtre->timeStampLastTriggerUp = longTimeStamp;
                    }
                    else if (filtre->trigState == TriggeredLow)
                    {
                        filtre->trigState = TriggeredHighAndLow;
                        filtre->timeStampLastTriggerUp = longTimeStamp;
                        //filtre->filtreMax->s_0 += _Q16_CONST_1 ;
                    }
                }
            }
        }
    }
    return updateOutput;
}

unsigned char FiltreMinMaxOrdre1CalculDecroissant(FiltreMinMaxOrdre1 * filtre, _Q16 input, long longTimeStamp)
{

    unsigned char updateOutput = 0;
    if (longTimeStamp > filtre->deadZoneDelay + filtre->timeStampLastTriggerDown)
        updateOutput = TRUE;

    //calcul de Vmin sur l'axe Z    
    if(updateOutput)
    {
        filtre->filtreMin->s_0 = FiltreOrdre1TpsReel(filtre->filtreMin, filtre->filtreMax->s_0);
    }
    
    if (input <= filtre->filtreMin->s_0)                                                  //ATTENTION COMPARAISON EN Q16
    {
        FiltreOrdre1SetOutput(filtre->filtreMin, _itofQ16(input));                          //Attention, la fonction attend un double..
        filtre->aMin = input;
        filtre->flagDecroissant = TRUE;
    }
    else
    {
        if (updateOutput == TRUE)
        {            
            if (filtre->flagDecroissant == TRUE)
            {
                //On vient de passer le min
                filtre->flagDecroissant = FALSE;
                filtre->aMin = filtre->filtreMin->s_0;
                //On teste si l'amplitude minimale a t respecte
                if(_Q16mac(filtre->filtreMax->s_0, _Q16_CONST_1, _Q16neg(filtre->aMin)) > filtre->amplitudeMin)
                {
                    if(filtre->trigStateStart==Waiting)
                    {
                        //On mmorise aussi la premire transition
                        filtre->trigStateStart=TriggeredLow;
                        filtre->trigState=TriggeredLow;
                        filtre->timeStampLastTriggerDown = longTimeStamp;
                    }
                    else if(filtre->trigState==TriggeredHigh)
                    {
                        filtre->trigState=TriggeredHighAndLow;
                        filtre->timeStampLastTriggerDown = longTimeStamp;
                    }
                }
            }
        }
    }
    return updateOutput;
}
#endif

void FiltreMinMaxOrdre1CalculCroissantdbl(FiltreMinMaxOrdre1dbl * filtre, double input, double amplitudeMin)
{
    //calcul de Vmax sur l'axe Z
    if (input >= filtre->aMaxMoy)          //ATTENTION COMPARAISON EN Q16
    {
        //VmaxMoyFiltree = Vmax = vit;
        FiltreOrdre1SetOutputdbl(filtre->filtreMax, input);      //Attention, la fonction attend un double..
        filtre->aMaxMoy = input;
        filtre->aMax = input;
        filtre->flagCroissant = TRUE;
    }
    else
    {
        if (filtre->flagCroissant == TRUE)
        {
            //On vient de passer le max
            filtre->flagCroissant = FALSE;
            filtre->aMax=filtre->aMaxMoy;
			//On teste si l'amplitude minimale a t respecte
			if(filtre->aMax-filtre->aMinMoy > amplitudeMin)
			{
				if(filtre->trigStateStart==Waiting)
				{
					//On mmorise aussi la premire transition
					filtre->trigStateStart=TriggeredHigh;
					filtre->trigState=TriggeredHigh;
				}
				else if(filtre->trigState==TriggeredLow)
					filtre->trigState=TriggeredHighAndLow;
			}
        }
        filtre->aMaxMoy = FiltreOrdre1TpsReeldbl(filtre->filtreMax, filtre->aMinMoy);
    }
}

void FiltreMinMaxOrdre1CalculDecroissantdbl(FiltreMinMaxOrdre1dbl * filtre, double vit, double amplitudeMin)
{
    //calcul de Vmin sur l'axe Z
    if (vit <= filtre->aMinMoy)                                                  //ATTENTION COMPARAISON EN Q16
    {
        FiltreOrdre1SetOutputdbl(filtre->filtreMin, vit);                          //Attention, la fonction attend un double..
        filtre->aMinMoy = vit;
        filtre->aMin = vit;
        filtre->flagDecroissant = TRUE;
    }
    else
    {
        if (filtre->flagDecroissant == TRUE)
        {
            //On vient de passer le min
            filtre->flagDecroissant = FALSE;
            filtre->aMin = filtre->aMinMoy;
			//On teste si l'amplitude minimale a t respecte
			if(filtre->aMaxMoy-filtre->aMin > amplitudeMin)
			{
				if(filtre->trigStateStart==Waiting)
				{
					//On mmorise aussi la premire transition
					filtre->trigStateStart=TriggeredLow;
					filtre->trigState=TriggeredLow;
				}
				else if(filtre->trigState==TriggeredHigh)
					filtre->trigState=TriggeredHighAndLow;
			}
        }
        //filtre->aMinMoy = FiltreOrdre1TpsReel(filtre->filtreMin, 0);
        filtre->aMinMoy = FiltreOrdre1TpsReeldbl(filtre->filtreMin, filtre->aMaxMoy);
    }
}

/*******************************************************************************
 * @fn      FiltreMinMaxOrdr2
 *
 * @brief   Fonction effectuant la detection d'enveloppe d'ordre 2.
 *          (detection du min et du max).
 *          Celle-ci actualise les parametres du filtre a partir de la valeure
 *          d'entre du filtre

 * @param   filtre - struct FiltreMinMaxOrdre2 - variable/objet filtre qui contiendra
 *          les infos/parametres du filtre.
 *          value - double - valeure d'entree du filtre (1 echantillon).
 *
 * @return  struct FiltreMinMaxOrdre2 - variable/objet filtre qui contiendra
 *          les infos/parametres du filtre actualis.
 *
 ******************************************************************************/
#ifdef USE_Q16
void FiltreMinMaxOrdre2Process(FiltreMinMaxOrdre2 * filtre, double value)
{
    if (value > filtre->aMaxMoy)
    {
        FiltreOrdre2SetOutput(filtre->filtreMax, value);
        filtre->aMaxMoy = value;
        filtre->flagCroissant = TRUE;
    }
    else
    {
        if (filtre->flagCroissant == TRUE)
        {
            //On vient de passer le max
            filtre->flagCroissant = FALSE;
            filtre->aMax = filtre->aMaxMoy;
            if (filtre->trigState == Waiting)
                filtre->trigState = TriggeredHigh;
            else if (filtre->trigState == TriggeredLow)
                filtre->trigState = TriggeredHighAndLow;
        }
        filtre->aMaxMoy = FiltreOrdre2TpsReel(filtre->filtreMax, filtre->aMinMoy);
        //filtre->aMaxMoy = 0;
    }

    if (value <= filtre->aMinMoy)
    {
        FiltreOrdre2SetOutput(filtre->filtreMin, value);
        filtre->aMinMoy = value;
        filtre->flagDecroissant = TRUE;
    }
    else
    {
        if (filtre->flagDecroissant == TRUE)
        {
            //On vient de passer le max
            filtre->flagDecroissant = FALSE;
            filtre->aMin = filtre->aMinMoy;
            if (filtre->trigState == Waiting)
                filtre->trigState = TriggeredLow;
            else if (filtre->trigState == TriggeredHigh)
                filtre->trigState = TriggeredHighAndLow;
        }
        filtre->aMinMoy = FiltreOrdre2TpsReel(filtre->filtreMin, filtre->aMaxMoy);
        //filtre.aMinMoy = 0;
    }
    //return filtre;
}


void FiltreMinMaxOrdre1ResetTrig(FiltreMinMaxOrdre1 * filtre)
{
    filtre->trigState = Waiting;
    filtre->trigStateStart = Waiting;
}
void FiltreMinMaxOrdre1ResetTrigdbl(FiltreMinMaxOrdre1dbl * filtre)
{
    filtre->trigState = Waiting;
    filtre->trigStateStart = Waiting;
}

void FiltreMinMaxOrdre1Reset(FiltreMinMaxOrdre1 * filtre)
{
    filtre->trigState = Waiting;
    filtre->trigStateStart = Waiting;
    filtre->aMax = 0;
    filtre->aMin = 0;
    filtre->flagCroissant = FALSE;
    filtre->flagDecroissant = FALSE;
    FiltreOrdre1SetOutput(filtre->filtreMax, 0);
    FiltreOrdre1SetOutput(filtre->filtreMin, 0);
}


/*******************************************************************************
 * Filtre Nombre de Repetitions
 ******************************************************************************/
void FiltreNbRepReset(FiltreNbRep * filtre)
{
    filtre->flag_axe = 'N';
    filtre->periodeRep = 0;
    filtre->timeStampMinMax = 0;
    filtre->timeStampMinMaxT_1 = 0;
    filtre->nbDeRepCumulees = 0;
    filtre->topValueDown = 0;
    filtre->topValueUp = 0;
    FiltreMinMaxOrdre1Reset(filtre->extremumLocal); // reset du trig de bas niveau
}


void FiltreNbRepInit(FiltreNbRep * filtre, FiltreMinMaxOrdre1 * filtreMinMax,
        FiltreOrdre1 * filtreLP1Min, FiltreOrdre1 * filtreLP1Max,
        double freqCoupure, double freqEch, long deadZoneDelay, double minTriggerValue)
{
    if(freqEch != 0)
    {
        //Filtre 1
        FiltrePasseBasButterworthOrdre1Init(filtreLP1Min, freqEch, freqCoupure);//filtre min
        FiltrePasseBasButterworthOrdre1Init(filtreLP1Max, freqEch, freqCoupure);//filtre max

        FiltreMinMaxOrdre1Init(filtreMinMax, freqEch, freqCoupure, deadZoneDelay, minTriggerValue);

        filtre->extremumLocal = filtreMinMax;
        filtre->extremumLocal->filtreMax = filtreLP1Max;
        filtre->extremumLocal->filtreMin = filtreLP1Min;
        filtre->timeStampMinMax = 0;
        filtre->timeStampMinMaxT_1 = 0;

        FiltreNbRepReset(filtre);
    }
}

unsigned char FiltreNbRepProcess(FiltreNbRep * filtre, _Q16Data * value, _Q16Data * valueTopMeasure, unsigned long timeStamp)
{
    unsigned char repEventDetected = FALSE;

    // calcul des extremums min et max pour les axes X, Y et Z
    if (FiltreMinMaxOrdre1CalculCroissant(filtre->extremumLocal, value->Value, timeStamp))
    {
        //On rcupre le max des vitesse en monte et descente
        //en dehors de la dead Zone
        if (valueTopMeasure->Value > filtre->topValueUp)
            filtre->topValueUp = valueTopMeasure->Value;
    }

    if (FiltreMinMaxOrdre1CalculDecroissant(filtre->extremumLocal, value->Value, timeStamp))
    {
        //On rcupre le max des vitesse en monte et descente
        //en dehors de la dead Zone
        if (valueTopMeasure->Value < filtre->topValueDown)
            filtre->topValueDown = valueTopMeasure->Value;
    }
    //******************************************************************//
    // detecteur des rep et freq de rep
    //******************************************************************//
    if(filtre->extremumLocal->trigState == TriggeredHighAndLow)
    {
        //On rinitialise le trigger
        FiltreMinMaxOrdre1ResetTrig(filtre->extremumLocal);

        filtre->nbDeRepCumulees++;
        repEventDetected = TRUE;

        //On mmorise l'instant de dclenchement
        filtre->timeStampMinMaxT_1 = filtre->timeStampMinMax;
        filtre->timeStampMinMax = timeStamp;

        //On calcule l'amplitude du mouvement
        filtre->amplitude = _Q16mac(filtre->extremumLocal->aMax, _Q16_CONST_1, _Q16neg(filtre->extremumLocal->aMin));

        // on a alors la periode entre deux repetitions
        if(filtre->timeStampMinMaxT_1 != 0)
        {
            filtre->periodeRep = (unsigned int)((filtre->timeStampMinMax-filtre->timeStampMinMaxT_1));
            if (filtre->periodeRep > 5000)
                filtre->periodeRep = 0;
        }

        filtre->topValueDownT_1 = filtre->topValueDown;
        filtre->topValueUpT_1 = filtre->topValueUp;

        filtre->topValueUp = 0;
        filtre->topValueDown = 0;
    }
    return repEventDetected;
}

/*******************************************************************************
 *   Filtre Idle Detector
 ******************************************************************************/
void _Q16IdleDetectorProcess(_Q16IdleDetector * idleDetector, _Q16Data * accel, unsigned long timestamp)
{
    //Si on est superieur a 0.1G ou inferieur a -0.1G et en mode Idle
    if(accel->Value > _Q16mac(idleDetector->idleAccelRefValue, _Q16_CONST_1, idleDetector->accelIdleSensitivity) || accel->Value< _Q16mac(idleDetector->idleAccelRefValue, _Q16_CONST_1,_Q16neg(idleDetector->accelIdleSensitivity)))
    {
        idleDetector->idleStartTime = timestamp;             //On reset le compteur
        idleDetector->idleAccelRefValue = accel->Value;
        if (idleDetector->flagAccelIdle == 1)
            idleDetector->flagAccelIdleChanged = 1;
        else
            idleDetector->flagAccelIdleChanged = 0;
        idleDetector->flagAccelIdle = 0;
    }
    //Si on atteint X ms en idle
    else if ((timestamp >= idleDetector->idleStartTime + idleDetector->accelIdleMinTime) && idleDetector->flagAccelIdle == 0)
    {
        idleDetector->flagAccelIdleChanged = 1;
        idleDetector->flagAccelIdle = 1;
        //On reset les filtres et les integrateurs
    }
    else
    {
        idleDetector->flagAccelIdleChanged = 0;
    }
}

unsigned char _Q16IdleDetectorIsIdle(_Q16IdleDetector * idleDetector)
{
    return idleDetector->flagAccelIdle;
}

unsigned char _Q16IdleDetectorStateHasChanged(_Q16IdleDetector * idleDetector)
{
    return idleDetector->flagAccelIdleChanged;
}

void _Q16IdleDetectorInit(_Q16IdleDetector * idleDetector, _Q16 sensitivity, unsigned int idleTime)
{
    idleDetector->accelIdleSensitivity = sensitivity;
    idleDetector->accelIdleMinTime = idleTime;
    idleDetector->flagAccelIdle = 0;
    idleDetector->flagAccelIdleChanged = 0;
    idleDetector->idleStartTime = 0;
}
#endif
/*******************************************************************************
 *
 ******************************************************************************/

/*static _Q16 idleAccelRefValue = 0;
BOOL flagAccelIdle = FALSE;
BOOL flagAccelIdleChanged = FALSE;
void _Q16IdleDetector(_Q16Data *accel, double fech)
{
    static long unsigned int idleTime;
    idleTime++;

    //Si on est superieur a 0.1G ou inferieur a -0.1G et en mode Idle
    if(accel->Value > _Q16mac(idleAccelRefValue, _Q16_CONST_1, _Q16_CONST_0_01G_MpS) || accel->Value< _Q16mac(idleAccelRefValue, _Q16_CONST_1,_Q16neg(_Q16_CONST_0_1G_MpS)))
    {
        idleTime=0;             //On reset le compteur
        idleAccelRefValue = accel->Value;
        if(flagAccelIdle==TRUE)
            flagAccelIdleChanged = TRUE;
        else
            flagAccelIdleChanged = FALSE;
        flagAccelIdle=FALSE;
    }
    
    //Si on atteint 1s en idle
    else if(idleTime >= (int)fech && flagAccelIdle==FALSE)
    {
        flagAccelIdleChanged = TRUE;
        flagAccelIdle=TRUE;
        //On reset les filtres et les integrateurs
    }

    else
    {
        flagAccelIdleChanged = FALSE;
    }
}

BOOL AccelIsIdle()
{
    return flagAccelIdle;
}
BOOL AccelIdleStateHasChanged()
{
    return flagAccelIdleChanged;
}
*/

#ifdef USE_Q16
_Q16 _Q16Square(_Q16 x, _Q16 divisor)
{
    return _Q16div(_Q16mac(x,x,NULLACCU),divisor);
}

static char sign;
void _Q16SignDetectorGet(_Q16 value)
{
    if(value>>31 == 1)
        sign=-1;
    else
        sign=1;
}
_Q16 _Q16SignDetectorSet(_Q16 value)
{
    if(sign<=0)
        return _Q16neg(value);
    else
        return value;
}
_Q16 _Q16Abs(_Q16 value)
{
    if(value>>31 == 1)
        return _Q16neg(value);
    else
        return value;
}
#endif

int CrossDetector(double signal1)
{
    static double oldEch;
    if(signal1>0 && oldEch>=0)
    {
        oldEch=signal1;
        return 1;
    }
    else if(signal1<0 && oldEch<=0)
    {
        oldEch=signal1;
        return 1;
    }
    else
    {
        oldEch=signal1;
        return 0;
    }
}
double Thresholddbl(double signal, Threshold *threshold)
{
    if(signal>=threshold->Value)
    {
        if(threshold->threshold_t_1==FALSE)
        {
            threshold->threshold_t_1=TRUE;
            return 1000000;
        }
        else
        {
            return 0;
        }
    }
    else
    {
        threshold->threshold_t_1=FALSE;
        return 0;
    }
}

#ifdef USE_ACCEL_RECALAGE
    /*******************************************************************************
     *      Fonctions de recalage
     * ****************************************************************************/
    FiltreOrdre1 ACCELPpRecalageFilter_LP1_accelX_LF;
    FiltreOrdre1 ACCELPpRecalageFilter_LP1_accelY_LF;
    FiltreOrdre1 ACCELPpRecalageFilter_LP1_accelZ_LF;

    static _Q16 thetaRecalage;
    static _Q16 phiRecalage;

    double ACCELPpfiltreFreqCoupureRecalageXYZ;

    _Q16DataXYZ ACCELRecaleAccel(_Q16DataXYZ accelNormalise)
    {
        //------------------------------------------------//
        //Recalage des data
        //------------------------------------------------//
        _Q16DataXYZ accelRecalVertical;
        ACCELPpFiltreRecalageProcess(accelNormalise, &accelRecalVertical);//Recalage sur Z
        return accelRecalVertical;
    }

    void ACCELPpFiltreRecalageInit()
    {
        ACCELPpfiltreFreqCoupureRecalageXYZ = 1.5;
        //Filtres de rcupration de la composante continue de l'acclration = orientation qui peut changer si le capteur bouge
        //Orientation par dfaut : y vers la tte, x latral cot droit, z vertical en haut:
        FiltrePasseBasButterworthOrdre1Init(&ACCELPpRecalageFilter_LP1_accelX_LF, ACCELRealSamplingFreq, ACCELPpfiltreFreqCoupureRecalageXYZ); //1.2
        FiltrePasseBasButterworthOrdre1Init(&ACCELPpRecalageFilter_LP1_accelY_LF, ACCELRealSamplingFreq, ACCELPpfiltreFreqCoupureRecalageXYZ); //0.6
        FiltrePasseBasButterworthOrdre1Init(&ACCELPpRecalageFilter_LP1_accelZ_LF, ACCELRealSamplingFreq, ACCELPpfiltreFreqCoupureRecalageXYZ); //3.0
        //FiltrePasseBasButterworthOrdre1Init(&MMA8453PpRecalageFilter_LP1_accelN_LF, freqEch, MMA8453PpfiltreFreqCoupureRecalageN); //3.0
    }

    void ACCELPpFiltreRecalageSetOutput(_Q16DataXYZ value)
    {
        FiltreOrdre1SetOutput(&ACCELPpRecalageFilter_LP1_accelX_LF,value.X);
        FiltreOrdre1SetOutput(&ACCELPpRecalageFilter_LP1_accelY_LF,value.Y);
        FiltreOrdre1SetOutput(&ACCELPpRecalageFilter_LP1_accelZ_LF,value.Z);
    }

    void ACCELPpFiltreRecalageProcess(_Q16DataXYZ accelQ16, _Q16DataXYZ * accelRecale_out)
    {
        //On rcupre la composante continue de l'acclration = orientation qui peut changer si le capteur bouge
        //Orientation par dfaut : y vers la tte, x latral cot droit, z vertical en haut

        _Q16 accelX_LF = FiltreOrdre1TpsReel(&ACCELPpRecalageFilter_LP1_accelX_LF, accelQ16.X);
        _Q16 accelY_LF = FiltreOrdre1TpsReel(&ACCELPpRecalageFilter_LP1_accelY_LF, accelQ16.Y);
        _Q16 accelZ_LF = FiltreOrdre1TpsReel(&ACCELPpRecalageFilter_LP1_accelZ_LF, accelQ16.Z);

        ACCELPpFiltreRecalageGetAngleParRapportVertical(accelX_LF, accelY_LF, accelZ_LF, &thetaRecalage, &phiRecalage);

        //On fait notre changement de base pour recaler le capteur en position
        static _Q16 result[3];
        //MMA8453PpFiltreRecalageSetBaseVerticale(accelXNormalise, accelYNormalise, accelZNormalise, *theta, *phi, result);
        ACCELPpFiltreRecalageSetBaseVerticale(accelQ16.X, accelQ16.Y, accelQ16.Z, thetaRecalage, phiRecalage, result);

        accelRecale_out->X = result[0];
        accelRecale_out->Y = result[1];
        accelRecale_out->Z = result[2];
    }

    void ACCELPpGetAngleRecalage(_Q16 *thetaQ16, _Q16 *phiQ16)
    {
        *thetaQ16 = thetaRecalage;
        *phiQ16 = (_Q16)phiRecalage;
    }

    void ACCELPpFiltreRecalageSetBaseVerticale(_Q16 accelX, _Q16 accelY, _Q16 accelZ, _Q16 theta, _Q16 phi, _Q16 result[])
    {
        _Q16 ct = _Q16cos(theta);
        _Q16 st = _Q16sin(theta);
        _Q16 cp = _Q16cos(phi);
        _Q16 sp = _Q16sin(phi);

        //Rotation selon oZ
        _Q16 X1 = _Q16mac(ct ,accelX,_Q16neg(_Q16mac(st ,accelZ,NULLACCU)));
        _Q16 Y1 = accelY;
        _Q16 Z1 = _Q16mac(ct,accelZ,_Q16mac(st,accelX,NULLACCU));

        //Rotation selon oY
        _Q16 X2 = X1;
        _Q16 Y2 = _Q16mac(_Q16neg(cp),Z1,_Q16mac(sp,Y1,NULLACCU));
        _Q16 Z2 = _Q16mac(cp ,Y1, _Q16mac(sp ,Z1,NULLACCU));

        result[0] = X2;
        result[1] = Y2;
        result[2] = Z2;
    }

    void ACCELPpFiltreRecalageGetAngleParRapportVertical(_Q16 accelX_LFQ16, _Q16 accelY_LFQ16, _Q16 accelZ_LFQ16, _Q16 *thetaQ16, _Q16 *phiQ16)
    {
        _Q16 accelN_LFQ16 = _Q16ftoi(sqrt(_itofQ16(_Q16mac(accelX_LFQ16,accelX_LFQ16,_Q16mac(accelY_LFQ16, accelY_LFQ16,_Q16mac(accelZ_LFQ16, accelZ_LFQ16,NULLACCU)) ) )));
        //_Q16 accelXY_LFQ16 = _Q16sqrt(_Q16mac(accelX_LFQ16, accelX_LFQ16,_Q16mac(accelY_LFQ16, accelY_LFQ16,NULLACCU)) );
       // _Q16 accelYZ_LFQ16 = _Q16sqrt(_Q16mac(accelY_LFQ16, accelY_LFQ16,_Q16mac(accelZ_LFQ16, accelZ_LFQ16,NULLACCU)) );
        _Q16 accelZX_LFQ16 = _Q16sqrt(_Q16mac(accelZ_LFQ16, accelZ_LFQ16,_Q16mac(accelX_LFQ16, accelX_LFQ16,NULLACCU)) );

         if (accelZX_LFQ16 != 0)
        {
            if (accelZ_LFQ16 >= 0 && accelX_LFQ16 >= 0)
                *thetaQ16 = _Q16neg(_Q16acos(_Q16div(accelZ_LFQ16 ,accelZX_LFQ16)));
            else if (accelZ_LFQ16 <= 0 && accelX_LFQ16 >= 0)
                *thetaQ16 = _Q16ftoi(PI) - _Q16acos(_Q16div(_Q16neg(accelZ_LFQ16),accelZX_LFQ16));
            else if (accelZ_LFQ16 <= 0 && accelX_LFQ16 <= 0)
                *thetaQ16 = _Q16ftoi(-PI) + _Q16acos(_Q16div(_Q16neg(accelZ_LFQ16),accelZX_LFQ16));
            else
                *thetaQ16 = _Q16neg(_Q16acos(_Q16div(accelZ_LFQ16 ,accelZX_LFQ16)));
        }
        else
            *thetaQ16 = 0;

        _Q16 nouvelleAccelZ_LF = _Q16mac(_Q16cos(*thetaQ16) ,accelZ_LFQ16,_Q16mac(_Q16sin(*thetaQ16) ,accelX_LFQ16,NULLACCU));
      //  _Q16 nouvelleAccelX_LF = _Q16mac(_Q16cos(*thetaQ16) ,accelX_LFQ16,_Q16neg(_Q16mac(_Q16sin(*thetaQ16) ,accelZ_LFQ16,NULLACCU)));

        if (nouvelleAccelZ_LF >= 0 && accelY_LFQ16 >= 0)
            *phiQ16 = _Q16asin(_Q16div(nouvelleAccelZ_LF ,accelN_LFQ16));
        else if (nouvelleAccelZ_LF <= 0 && accelY_LFQ16 <= 0)
            *phiQ16 = _Q16ftoi(-PI) + _Q16asin(_Q16div(_Q16neg(nouvelleAccelZ_LF) ,accelN_LFQ16));
        else if (nouvelleAccelZ_LF >= 0 && accelY_LFQ16 <= 0)
            *phiQ16 = _Q16ftoi(PI) - _Q16asin(_Q16div(nouvelleAccelZ_LF ,accelN_LFQ16));
        else
            *phiQ16 = _Q16neg(_Q16asin(_Q16div(_Q16neg(nouvelleAccelZ_LF) ,accelN_LFQ16)));
    }
#endif


#ifdef USE_ACCEL_FILTER
void AccelPpFiltersInit()
{
    #ifdef USE_ACCEL_RECALAGE
        ACCELPpFiltreRecalageInit();
        #ifdef USE_ENERGIE_FILTER
        ACCELPpAccelRMSFiltersInit();
        #endif

        #ifdef USE_ACCEL_FILTER
            //Initialisation avec des parametres fitness
            ACCELPpInitFilters();
            nbSerieDone_X=1;              //On reset le nombre de serie
            nbSerieDone_Y=1;              //On reset le nombre de serie
            nbSerieDone_Z=1;              //On reset le nombre de serie
        #endif
    #endif
    #ifdef USE_EQUILIBRE
        ACCELPpInitFilters_Equilibre();
    #endif
}


// Filtres Energie
FiltreOrdre1 ACCELPpFilterAccelXHF;
FiltreOrdre1 ACCELPpFilterAccelYHF;
FiltreOrdre1 ACCELPpFilterAccelZHF;

#ifdef USE_ENERGIE_FILTER
FiltreOrdre1 ACCELPpFilterEnergieX;
FiltreOrdre1 ACCELPpFilterEnergieY;
FiltreOrdre1 ACCELPpFilterEnergieZ;

_Q16DataXYZ ACCELPpAccelRMS(_Q16DataXYZ accelRecalVertical)
{
    static _Q16DataXYZ ACCELPpAccelRMS;
    _Q16 accelZHF;

    //On filtre la composante continue en Z
    accelZHF = FiltreOrdre1TpsReel(&ACCELPpFilterAccelZHF, accelRecalVertical.Z);

    //On calcule la valeur RMS glissante
    ACCELPpAccelRMS.Z = _Q16sqrt(FiltreOrdre1TpsReel(&ACCELPpFilterEnergieZ, _Q16mac(accelZHF, accelZHF, NULLACCU)));
    ACCELPpAccelRMS.X = _Q16sqrt(FiltreOrdre1TpsReel(&ACCELPpFilterEnergieX, _Q16mac(accelRecalVertical.X, accelRecalVertical.X, NULLACCU)));
    ACCELPpAccelRMS.Y = _Q16sqrt(FiltreOrdre1TpsReel(&ACCELPpFilterEnergieY, _Q16mac(accelRecalVertical.Y, accelRecalVertical.Y, NULLACCU)));

    return ACCELPpAccelRMS;
 }


void ACCELPpAccelRMSFiltersInit()
{
    FiltrePasseHautButterworthOrdre1Init(&ACCELPpFilterAccelZHF, ACCELRealSamplingFreq, 0.5*6.28);
    FiltrePasseBasButterworthOrdre1Init(&ACCELPpFilterEnergieX, ACCELRealSamplingFreq, 0.2*6.28);
    FiltrePasseBasButterworthOrdre1Init(&ACCELPpFilterEnergieY, ACCELRealSamplingFreq, 0.1*6.28);
    FiltrePasseBasButterworthOrdre1Init(&ACCELPpFilterEnergieZ, ACCELRealSamplingFreq, 0.5*6.28);
}
#endif
#ifdef USE_ACCEL_FILTER 
/*******************************************************************************
 *      Post Processing Variables
 * ****************************************************************************/
BOOL ACCELPpUseRecalage = TRUE;
BOOL ACCELPpUseEnergy = TRUE;
BOOL ACCELPpUsePosition = TRUE;

_Q16DataXYZ ACCELPpEnergie;

int ACCELPpcounterUpdateFreqFilter;

unsigned char ACCELPpMachineExerciceType;

_Q16IdleDetector ACCELPpIdleDetector_Z;
//#ifdef USE_ACCEL_3D_FILTER
//_Q16IdleDetector ACCELPpIdleDetector_X;
//_Q16IdleDetector ACCELPpIdleDetector_Y;
//#endif


//FIltres Nb de Rptitions
//Axe Z
FiltreNbRep ACCELPpNbRepFilter_NbRep1_Z;
FiltreOrdre1 ACCELPpNbRepFilter_LP1_Max1_Z;
FiltreOrdre1 ACCELPpNbRepFilter_LP1_Min1_Z;
FiltreMinMaxOrdre1 ACCELPpNbRepFilter_MinMax_1_Z;
#ifdef USE_ACCEL_3D_FILTER
//Axe X
FiltreNbRep ACCELPpNbRepFilter_NbRep1_X;
FiltreOrdre1 ACCELPpNbRepFilter_LP1_Max1_X;
FiltreOrdre1 ACCELPpNbRepFilter_LP1_Min1_X;
FiltreMinMaxOrdre1 ACCELPpNbRepFilter_MinMax_1_X;
//Axe Y
FiltreNbRep ACCELPpNbRepFilter_NbRep1_Y;
FiltreOrdre1 ACCELPpNbRepFilter_LP1_Max1_Y;
FiltreOrdre1 ACCELPpNbRepFilter_LP1_Min1_Y;
FiltreMinMaxOrdre1 ACCELPpNbRepFilter_MinMax_1_Y;
#endif



//Filtres vitesse et position
//Axe Z
_Q16Integrateur ACCELPpIntegrateurVitesse_Z;
FiltreOrdre1 ACCELPpFiltreVitesse_Z;
_Q16Integrateur ACCELPpIntegrateurPosition_Z;
FiltreOrdre1 ACCELPpFiltrePosition_Z;
#ifdef USE_ACCEL_3D_FILTER
//Axe Y
_Q16Integrateur ACCELPpIntegrateurVitesse_Y;
FiltreOrdre1 ACCELPpFiltreVitesse_Y;
_Q16Integrateur ACCELPpIntegrateurPosition_Y;
FiltreOrdre1 ACCELPpFiltrePosition_Y;
//Axe X
_Q16Integrateur ACCELPpIntegrateurVitesse_X;
FiltreOrdre1 ACCELPpFiltreVitesse_X;
_Q16Integrateur ACCELPpIntegrateurPosition_X;
FiltreOrdre1 ACCELPpFiltrePosition_X;
#endif



#ifdef USE_ACCEL_FILTER
FiltreOrdre1 filtreAccelN_HP;
FiltreOrdre1 filtreEnergie;
Integrateur actimetre;
FiltreMinMaxOrdre2 filtreAmin;
#endif

_Q16DataXYZ ACCELPpRepetionPosition(_Q16DataXYZ accelNormalise, _Q16DataXYZ accelRecalVertical)
{
    //------------------------------------------------//
    //Calcul de vitesse et de position
    //------------------------------------------------//

    //Detection d'idle de l'accel (pour eviter la derive des integrateurs)
    _Q16 squarNorm = _Q16mac(accelRecalVertical.X,accelRecalVertical.X,_Q16mac(accelRecalVertical.Y,accelRecalVertical.Y,_Q16mac(accelRecalVertical.Z,accelRecalVertical.Z,NULLACCU)));
    _Q16IdleDetectorProcess( &ACCELPpIdleDetector_Z, (_Q16Data*)&squarNorm, (unsigned int)g_longTimeStamp);
//    #ifdef USE_ACCEL_3D_FILTER
//    _Q16IdleDetectorProcess( &ACCELPpIdleDetector_X, (_Q16Data*)&accelRecalVertical.X, (unsigned int)g_longTimeStamp);
//    _Q16IdleDetectorProcess( &ACCELPpIdleDetector_Y, (_Q16Data*)&accelRecalVertical.Y, (unsigned int)g_longTimeStamp);
//    #endif

    if(_Q16IdleDetectorIsIdle(&ACCELPpIdleDetector_Z))
    {
        #ifdef USE_ACCEL_3D_FILTER
        if(ACCELPpNbRepFilter_NbRep1_Z.nbDeRepCumulees>=1 || ACCELPpNbRepFilter_NbRep1_X.nbDeRepCumulees>=1 || ACCELPpNbRepFilter_NbRep1_Y.nbDeRepCumulees>=1)
        #else
        if(ACCELPpNbRepFilter_NbRep1_Z.nbDeRepCumulees>=1)
        #endif
        {
            //On incremente le nombre de Serie.
            nbSerieDone_Z++;
            #ifdef USE_ACCEL_3D_FILTER
                nbSerieDone_X++;
                nbSerieDone_Y++;
            #endif
            //On reset le filtre de nb de rptitions-> remet le nbRep  0
            ACCELPpNbRepFilter_NbRep1_Z.nbDeRepCumulees=0;
            #ifdef USE_ACCEL_3D_FILTER
                ACCELPpNbRepFilter_NbRep1_X.nbDeRepCumulees=0;
                ACCELPpNbRepFilter_NbRep1_Y.nbDeRepCumulees=0;
            #endif
            if(flagAccelPostProcessing.usePolling == 0)
            {
                //Envoi d'une trame d'indication de nouvelle srie
                SendTrameNbRepToRadio(AXE_Z,&ACCELPpNbRepFilter_NbRep1_Z);
                #if defined( USE_ACCEL_3D_FILTER) && defined(USE_3D_TRAME)
                    SendTrameNbRepToRadio(AXE_X,&ACCELPpNbRepFilter_NbRep1_X);
                    SendTrameNbRepToRadio(AXE_Y,&ACCELPpNbRepFilter_NbRep1_Y);
                #endif
            }
        }

        //On reset l'integrateur de la vitesse non filtree
        _Q16IntegrateurReset(&ACCELPpIntegrateurVitesse_Z);
        FiltreOrdre1SetOutput(&ACCELPpFiltreVitesse_Z, 0);
        #ifdef USE_ACCEL_3D_FILTER
            _Q16IntegrateurReset(&ACCELPpIntegrateurVitesse_X);
            FiltreOrdre1SetOutput(&ACCELPpFiltreVitesse_X, 0);
            _Q16IntegrateurReset(&ACCELPpIntegrateurVitesse_Y);
            FiltreOrdre1SetOutput(&ACCELPpFiltreVitesse_Y, 0);
        #endif

        //On reset l'integrateur de la position
        _Q16IntegrateurReset(&ACCELPpIntegrateurPosition_Z);
        #ifdef USE_ACCEL_3D_FILTER
            _Q16IntegrateurReset(&ACCELPpIntegrateurPosition_X);
            _Q16IntegrateurReset(&ACCELPpIntegrateurPosition_Y);
        #endif

        //On reset le filtre passe haut de la position
        FiltreOrdre1SetOutput(&ACCELPpFiltrePosition_Z, 0);
        #ifdef USE_ACCEL_3D_FILTER
            FiltreOrdre1SetOutput(&ACCELPpFiltrePosition_X, 0);
            FiltreOrdre1SetOutput(&ACCELPpFiltrePosition_Y, 0);
        #endif

        //On reset le filtre de nb de rptitions
        FiltreNbRepReset(&ACCELPpNbRepFilter_NbRep1_Z);
        #ifdef USE_ACCEL_3D_FILTER
            FiltreNbRepReset(&ACCELPpNbRepFilter_NbRep1_X);
            FiltreNbRepReset(&ACCELPpNbRepFilter_NbRep1_Y);
        #endif
    }
//    #ifdef USE_ACCEL_3D_FILTER
//    if(_Q16IdleDetectorIsIdle(&ACCELPpIdleDetector_X))
//    {
//        if(ACCELPpNbRepFilter_NbRep1_X.nbDeRepCumulees>=1)
//        {
//            //On incremente le nombre de Serie.
//            nbSerieDone_X++;
//            //On reset le filtre de nb de rptitions-> remet le nbRep  0
//            ACCELPpNbRepFilter_NbRep1_X.nbDeRepCumulees=0;
//
//            if(flagAccelPostProcessing.usePolling == 0)
//            {
//                //Envoi d'une trame d'indication de nouvelle srie
//                SendTrameNbRep(AXE_X,&ACCELPpNbRepFilter_NbRep1_X);
//            }
//        }
//
//        //On reset l'integrateur de la vitesse non filtree
//        _Q16IntegrateurReset(&ACCELPpIntegrateurVitesse_X);
//        FiltreOrdre1SetOutput(&ACCELPpFiltreVitesse_X, 0);
//
//        //On reset l'integrateur de la position
//        _Q16IntegrateurReset(&ACCELPpIntegrateurPosition_X);
//
//        //On reset le filtre passe haut de la position
//        FiltreOrdre1SetOutput(&ACCELPpFiltrePosition_X, 0);
//
//        //On reset le filtre de nb de rptitions
//        FiltreNbRepReset(&ACCELPpNbRepFilter_NbRep1_X);
//    }
//
//
//    if(_Q16IdleDetectorIsIdle(&ACCELPpIdleDetector_Y))
//    {
//        if(ACCELPpNbRepFilter_NbRep1_Y.nbDeRepCumulees>=1)
//        {
//            //On incremente le nombre de Serie.
//            nbSerieDone_Y++;
//            //On reset le filtre de nb de rptitions-> remet le nbRep  0
//            ACCELPpNbRepFilter_NbRep1_Y.nbDeRepCumulees=0;
//
//            if(flagAccelPostProcessing.usePolling == 0)
//            {
//                //Envoi d'une trame d'indication de nouvelle srie
//                SendTrameNbRep(AXE_Y,&ACCELPpNbRepFilter_NbRep1_Y);
//            }
//        }
//
//        //On reset l'integrateur de la vitesse non filtree
//        _Q16IntegrateurReset(&ACCELPpIntegrateurVitesse_Y);
//        FiltreOrdre1SetOutput(&ACCELPpFiltreVitesse_Y, 0);
//
//        //On reset l'integrateur de la position
//        _Q16IntegrateurReset(&ACCELPpIntegrateurPosition_Y);
//
//        //On reset le filtre passe haut de la position
//        FiltreOrdre1SetOutput(&ACCELPpFiltrePosition_Y, 0);
//
//        //On reset le filtre de nb de rptitions
//        FiltreNbRepReset(&ACCELPpNbRepFilter_NbRep1_Y);
//    }
//    #endif
    
    //Filtrage passe-haut de l'acclration
    _Q16 accelZHF = FiltreOrdre1TpsReel(&ACCELPpFilterAccelZHF, accelRecalVertical.Z);
    #ifdef USE_ACCEL_3D_FILTER
    _Q16 accelXHF = FiltreOrdre1TpsReel(&ACCELPpFilterAccelXHF, accelRecalVertical.X);
    _Q16 accelYHF = FiltreOrdre1TpsReel(&ACCELPpFilterAccelYHF, accelRecalVertical.Y);
    #endif

    //Dtermination de la vitesse
    _Q16 vitesseZ = _Q16Integrate(&ACCELPpIntegrateurVitesse_Z, accelZHF);
    _Q16 vitesseZHF = FiltreOrdre1TpsReel(&ACCELPpFiltreVitesse_Z, vitesseZ);
    #ifdef USE_ACCEL_3D_FILTER
    _Q16 vitesseX = _Q16Integrate(&ACCELPpIntegrateurVitesse_X, accelXHF);
    _Q16 vitesseXHF = FiltreOrdre1TpsReel(&ACCELPpFiltreVitesse_X, vitesseX);
    _Q16 vitesseY = _Q16Integrate(&ACCELPpIntegrateurVitesse_Y, accelYHF);
    _Q16 vitesseYHF = FiltreOrdre1TpsReel(&ACCELPpFiltreVitesse_Y, vitesseY);
    #endif

    //Dtermination de la position
    _Q16 positionZ = _Q16Integrate(&ACCELPpIntegrateurPosition_Z, vitesseZHF);
    _Q16 positionZHF = FiltreOrdre1TpsReel(&ACCELPpFiltrePosition_Z, positionZ);
    #ifdef USE_ACCEL_3D_FILTER
     _Q16 positionX = _Q16Integrate(&ACCELPpIntegrateurPosition_X, vitesseXHF);
    _Q16 positionXHF = FiltreOrdre1TpsReel(&ACCELPpFiltrePosition_X, positionX);
     _Q16 positionY = _Q16Integrate(&ACCELPpIntegrateurPosition_Y, vitesseYHF);
    _Q16 positionYHF = FiltreOrdre1TpsReel(&ACCELPpFiltrePosition_Y, positionY);
    #endif

    if(ACCELPpUsePosition)//Calcule du nombre de rep en position
    {
        BOOL repetitionEventZ = FiltreNbRepProcess(&ACCELPpNbRepFilter_NbRep1_Z, (_Q16Data*)&positionZHF, (_Q16Data*)&vitesseZHF, g_longTimeStamp);
        #ifdef USE_ACCEL_3D_FILTER
        BOOL repetitionEventX = FiltreNbRepProcess(&ACCELPpNbRepFilter_NbRep1_X, (_Q16Data*)&positionXHF, (_Q16Data*)&vitesseXHF, g_longTimeStamp);
        BOOL repetitionEventY = FiltreNbRepProcess(&ACCELPpNbRepFilter_NbRep1_Y, (_Q16Data*)&positionYHF, (_Q16Data*)&vitesseYHF, g_longTimeStamp);
        #endif
        if(repetitionEventZ == TRUE)
        {
            if(flagAccelPostProcessing.usePolling == 0)
                if(ACCELPpNbRepFilter_NbRep1_Z.nbDeRepCumulees >= ACCELPpNbRepFilter_NbRep1_Y.nbDeRepCumulees && ACCELPpNbRepFilter_NbRep1_Z.nbDeRepCumulees >= ACCELPpNbRepFilter_NbRep1_X.nbDeRepCumulees)
                    SendTrameNbRepToRadio(AXE_Z,&ACCELPpNbRepFilter_NbRep1_Z);
        }
        #ifdef USE_ACCEL_3D_FILTER
        if(repetitionEventX == TRUE)
        {
            if(flagAccelPostProcessing.usePolling == 0)
                if(ACCELPpNbRepFilter_NbRep1_X.nbDeRepCumulees >= ACCELPpNbRepFilter_NbRep1_Y.nbDeRepCumulees && ACCELPpNbRepFilter_NbRep1_X.nbDeRepCumulees > ACCELPpNbRepFilter_NbRep1_Z.nbDeRepCumulees)
                    SendTrameNbRepToRadio(AXE_X,&ACCELPpNbRepFilter_NbRep1_X);
        }
        if(repetitionEventY == TRUE)
        {
            if(flagAccelPostProcessing.usePolling == 0)
                if(ACCELPpNbRepFilter_NbRep1_Y.nbDeRepCumulees > ACCELPpNbRepFilter_NbRep1_Z.nbDeRepCumulees && ACCELPpNbRepFilter_NbRep1_Y.nbDeRepCumulees > ACCELPpNbRepFilter_NbRep1_X.nbDeRepCumulees)
                    SendTrameNbRepToRadio(AXE_Y,&ACCELPpNbRepFilter_NbRep1_Y);
        }
        #endif
    }

    static _Q16DataXYZ output;   
    output.X = _Q16mac(accelRecalVertical.X,_Q16_CONST_100,NULLACCU);
    output.Y = _Q16mac(accelRecalVertical.Y,_Q16_CONST_100,NULLACCU);
    output.Z = _Q16mac(accelRecalVertical.Z,_Q16_CONST_100,NULLACCU);

    return output;
}

void ACCELPpInitFilters()
{
    //appel permettant une config rapide en fonction des usages
    ACCELPpInitFilters_Fitness();
}

//Paramtres par dfaut
double ACCELPpfiltreFreqCoupurePos = 1.0;
double ACCELPpfiltreFreqCoupureVit = 2.0;
double ACCELPpfiltreFreqCoupureAccel = 2.0;

double ACCELPpfiltreFreqCoupureNbRep = 0.5;
long ACCELPpFiltreMinMaxDeadZoneDelay = 400;
double ACCELPpFiltreMinMaxTriggerMinValue = 0.1;

void ACCELPpFiltreNbRepSetParameters(double freqCoupure, long deadZoneDelay, double minTriggerValue)
{
    ACCELPpfiltreFreqCoupureNbRep = freqCoupure;
    ACCELPpFiltreMinMaxDeadZoneDelay = deadZoneDelay;
    ACCELPpFiltreMinMaxTriggerMinValue = minTriggerValue;
}

void ACCELPpInitFilters_Fitness()
{
    //On valide si la frquence est diffrente de 0
    if(ACCELRealSamplingFreq!=0)
    {
        //Filtres nergie
        #ifdef USE_ENERGIE_FILTER
            const double ACCELPpfiltreFreqCoupureEnergie = 0.2;
            FiltrePasseBasButterworthOrdre1Init(&ACCELPpFilterEnergieZ, ACCELRealSamplingFreq, ACCELPpfiltreFreqCoupureEnergie);
            #ifdef USE_ACCEL_3D_FILTER
            FiltrePasseBasButterworthOrdre1Init(&ACCELPpFilterEnergieX, ACCELRealSamplingFreq, ACCELPpfiltreFreqCoupureEnergie);
            FiltrePasseBasButterworthOrdre1Init(&ACCELPpFilterEnergieY, ACCELRealSamplingFreq, ACCELPpfiltreFreqCoupureEnergie);
            #endif
        #endif

        //Filtre IdleDetector
        //_Q16IdleDetectorInit(&ACCELPpIdleDetector_Z, _Q16ftoi(12), 3000);        //1.5 sens
        _Q16IdleDetectorInit(&ACCELPpIdleDetector_Z, _Q16ftoi(24), 2500);        //1.5 sens
    //    #ifdef USE_ACCEL_3D_FILTER
    //    _Q16IdleDetectorInit(&ACCELPpIdleDetector_X, _Q16ftoi(1.5), 3000);
    //    _Q16IdleDetectorInit(&ACCELPpIdleDetector_Y, _Q16ftoi(1.5), 3000);
    //    #endif

        //Filtres Allures

        //Filtre acclration
        FiltrePasseHautButterworthOrdre1Init(&ACCELPpFilterAccelZHF, ACCELRealSamplingFreq, ACCELPpfiltreFreqCoupureAccel);
        #ifdef USE_ACCEL_3D_FILTER
        FiltrePasseHautButterworthOrdre1Init(&ACCELPpFilterAccelXHF, ACCELRealSamplingFreq, ACCELPpfiltreFreqCoupureAccel);
        FiltrePasseHautButterworthOrdre1Init(&ACCELPpFilterAccelYHF, ACCELRealSamplingFreq, ACCELPpfiltreFreqCoupureAccel);
        #endif


        //Filtres vitesse
        //Axe Z
        _Q16IntegrateurInit(&ACCELPpIntegrateurVitesse_Z, ACCELRealSamplingFreq);
        FiltrePasseHautButterworthOrdre1Init(&ACCELPpFiltreVitesse_Z, ACCELRealSamplingFreq, ACCELPpfiltreFreqCoupureVit);
        #ifdef USE_ACCEL_3D_FILTER
        _Q16IntegrateurInit(&ACCELPpIntegrateurVitesse_X, ACCELRealSamplingFreq);
        _Q16IntegrateurInit(&ACCELPpIntegrateurVitesse_Y, ACCELRealSamplingFreq);
        FiltrePasseHautButterworthOrdre1Init(&ACCELPpFiltreVitesse_X, ACCELRealSamplingFreq, ACCELPpfiltreFreqCoupureVit);
        FiltrePasseHautButterworthOrdre1Init(&ACCELPpFiltreVitesse_Y, ACCELRealSamplingFreq, ACCELPpfiltreFreqCoupureVit);
        #endif



        //Filtres Positions
        _Q16IntegrateurInit(&ACCELPpIntegrateurPosition_Z, ACCELRealSamplingFreq);
        FiltrePasseHautButterworthOrdre1Init(&ACCELPpFiltrePosition_Z, ACCELRealSamplingFreq, ACCELPpfiltreFreqCoupurePos);
        #ifdef USE_ACCEL_3D_FILTER
        _Q16IntegrateurInit(&ACCELPpIntegrateurPosition_X, ACCELRealSamplingFreq);
        FiltrePasseHautButterworthOrdre1Init(&ACCELPpFiltrePosition_X, ACCELRealSamplingFreq, ACCELPpfiltreFreqCoupurePos);
        _Q16IntegrateurInit(&ACCELPpIntegrateurPosition_Y, ACCELRealSamplingFreq);
        FiltrePasseHautButterworthOrdre1Init(&ACCELPpFiltrePosition_Y, ACCELRealSamplingFreq, ACCELPpfiltreFreqCoupurePos);
        #endif


        //FIltre NbRep
        FiltreNbRepInit(&ACCELPpNbRepFilter_NbRep1_Z, &ACCELPpNbRepFilter_MinMax_1_Z,
            &ACCELPpNbRepFilter_LP1_Min1_Z, &ACCELPpNbRepFilter_LP1_Max1_Z,
            ACCELPpfiltreFreqCoupureNbRep, ACCELRealSamplingFreq,
            ACCELPpFiltreMinMaxDeadZoneDelay, ACCELPpFiltreMinMaxTriggerMinValue);
        #ifdef USE_ACCEL_3D_FILTER
        FiltreNbRepInit(&ACCELPpNbRepFilter_NbRep1_X, &ACCELPpNbRepFilter_MinMax_1_X,
            &ACCELPpNbRepFilter_LP1_Min1_X, &ACCELPpNbRepFilter_LP1_Max1_X,
            ACCELPpfiltreFreqCoupureNbRep, ACCELRealSamplingFreq,
            ACCELPpFiltreMinMaxDeadZoneDelay, ACCELPpFiltreMinMaxTriggerMinValue);

        FiltreNbRepInit(&ACCELPpNbRepFilter_NbRep1_Y, &ACCELPpNbRepFilter_MinMax_1_Y,
            &ACCELPpNbRepFilter_LP1_Min1_Y, &ACCELPpNbRepFilter_LP1_Max1_Y,
            ACCELPpfiltreFreqCoupureNbRep, ACCELRealSamplingFreq,
            ACCELPpFiltreMinMaxDeadZoneDelay, ACCELPpFiltreMinMaxTriggerMinValue);
        #endif
    }
}

#endif

#endif  //ifdef USE_ACCEL_RECALAGE


#ifdef USE_EQUILIBRE
#ifndef ABS_Q16
#define ABS_Q16(x) ((x>0) ? x : _Q16neg(x))
#endif
#define EQUI_IN_DBL
//#define USE_FILTRE_EQUI_ORDRE_2
#define FILTRE_EQUI_LP_FC 4*2*PI//5
#define FILTRE_EQUI_HP_FC 0.3*2*PI

#ifdef EQUI_IN_DBL
double RMSMedioLateralVerticalPhase = 0;
double MeanJerkTotalPhase = 0;
double RMSJerkVerticalPhase = 0;

_Q16 APF_1=0;
_Q16 MLF_1=0;
_Q16 VF_1=0;

double SPML = 0;
double SPV = 0;
double SPAPML = 0;
double SPMLV = 0;
double SPT = 0;
double SAAML = 0;
double SAAV = 0;
double SAAAPML = 0;
double SAAMLV = 0;
double SAAT = 0;

#else
_Q16 RMSMedioLateralVerticalPhase = 0;
_Q16 MeanJerkTotalPhase = 0;
_Q16 RMSJerkVerticalPhase = 0;

_Q16 APF_1=0;
_Q16 MLF_1=0;
_Q16 VF_1=0;

_Q16 SPML = 0;
_Q16 SPV = 0;
_Q16 SPAPML = 0;
_Q16 SPMLV = 0;
_Q16 SPT = 0;
_Q16 SAAML = 0;
_Q16 SAAV = 0;
_Q16 SAAAPML = 0;
_Q16 SAAMLV = 0;
_Q16 SAAT = 0;

#endif
#ifdef USE_FILTRE_EQUI_ORDRE_2
FiltreOrdre2 filtrePasseHautAnteroPosterieur;
FiltreOrdre2 filtrePasseHautMedioLateral;
FiltreOrdre2 filtrePasseHautVertical;

FiltreOrdre2 filtrePasseBasAnteroPosterieur;
FiltreOrdre2 filtrePasseBasMedioLateral;
FiltreOrdre2 filtrePasseBasVertical;
#else
FiltreOrdre1 filtrePasseHautAnteroPosterieur;
FiltreOrdre1 filtrePasseHautMedioLateral;
FiltreOrdre1 filtrePasseHautVertical;

FiltreOrdre1 filtrePasseBasAnteroPosterieur;
FiltreOrdre1 filtrePasseBasMedioLateral;
FiltreOrdre1 filtrePasseBasVertical;

FiltreOrdre1 filtrePasseHautAnteroPosterieur_2;
FiltreOrdre1 filtrePasseHautMedioLateral_2;
FiltreOrdre1 filtrePasseHautVertical_2;

FiltreOrdre1 filtrePasseBasAnteroPosterieur_2;
FiltreOrdre1 filtrePasseBasMedioLateral_2;
FiltreOrdre1 filtrePasseBasVertical_2;
#endif

//static volatile double accX2;
//static volatile double accY2;
//static volatile double accZ2;
//static volatile double accX3;
//static volatile double accY3;
//static volatile double accZ3;
_Q16DataXYZ ACCELPpEquilibreFiltre(_Q16DataXYZ accelNormalise)
{
    //DEBUG ACCEL NORMALISE
//    accX2=_itofQ16(accelNormalise.X);
//    accY2=_itofQ16(accelNormalise.Y);
//    accZ2=_itofQ16(accelNormalise.Z);

//    accelNormalise.Z=_Q16mac(accelNormalise.Z,_Q16power(_Q16_CONST_100, _Q16neg(_Q16_CONST_1)), NULLACCU);
//    accelNormalise.Y=_Q16mac(accelNormalise.Y,_Q16power(_Q16_CONST_100, _Q16neg(_Q16_CONST_1)), NULLACCU);
//    accelNormalise.X=_Q16mac(accelNormalise.X,_Q16power(_Q16_CONST_100, _Q16neg(_Q16_CONST_1)), NULLACCU);
    #ifdef USE_FILTRE_EQUI_ORDRE_2
        _Q16 accelApHF = FiltreOrdre2TpsReel(&filtrePasseHautAnteroPosterieur, accelNormalise.Z);       //Accel Antero Posterieur
        _Q16 accelMlHF = FiltreOrdre2TpsReel(&filtrePasseHautMedioLateral, accelNormalise.Y);           //Accel Medio Lateral
        _Q16 accelVHF = FiltreOrdre2TpsReel(&filtrePasseHautVertical, accelNormalise.X);                //Accel vertical
        accX3=_itofQ16(accelApHF);
        accY3=_itofQ16(accelMlHF);
        accZ3=_itofQ16(accelVHF);
        _Q16 accelApLF = FiltreOrdre2TpsReel(&filtrePasseBasAnteroPosterieur, accelApHF);       //Accel Antero Posterieur
        _Q16 accelMlLF = FiltreOrdre2TpsReel(&filtrePasseBasMedioLateral, accelMlHF);           //Accel Medio Lateral
        _Q16 accelVLF = FiltreOrdre2TpsReel(&filtrePasseBasVertical, accelVHF);                //Accel vertical
    #else
        //======================First Pass HP ================================//
        _Q16 accelApHF = FiltreOrdre1TpsReel(&filtrePasseHautAnteroPosterieur, accelNormalise.Z);       //Accel Antero Posterieur
        _Q16 accelMlHF = FiltreOrdre1TpsReel(&filtrePasseHautMedioLateral, accelNormalise.Y);           //Accel Medio Lateral
        _Q16 accelVHF = FiltreOrdre1TpsReel(&filtrePasseHautVertical, accelNormalise.X);                //Accel vertical

        //======================Second Pass HP ===============================//
        accelApHF = FiltreOrdre1TpsReel(&filtrePasseHautAnteroPosterieur_2, accelApHF);       //Accel Antero Posterieur
        accelMlHF = FiltreOrdre1TpsReel(&filtrePasseHautMedioLateral_2, accelMlHF);           //Accel Medio Lateral
        accelVHF = FiltreOrdre1TpsReel(&filtrePasseHautVertical_2, accelVHF);                //Accel vertical

        //======================First Pass LP ================================//
        _Q16 accelApLF = FiltreOrdre1TpsReel(&filtrePasseBasAnteroPosterieur, accelApHF);       //Accel Antero Posterieur
        _Q16 accelMlLF = FiltreOrdre1TpsReel(&filtrePasseBasMedioLateral, accelMlHF);           //Accel Medio Lateral
        _Q16 accelVLF = FiltreOrdre1TpsReel(&filtrePasseBasVertical, accelVHF);                //Accel vertical

        //======================Second Pass LP ===============================//
        accelApLF = FiltreOrdre1TpsReel(&filtrePasseBasAnteroPosterieur_2, accelApLF);       //Accel Antero Posterieur
        accelMlLF = FiltreOrdre1TpsReel(&filtrePasseBasMedioLateral_2, accelMlLF);           //Accel Medio Lateral
        accelVLF = FiltreOrdre1TpsReel(&filtrePasseBasVertical_2, accelVLF);                //Accel vertical

    #endif
    _Q16DataXYZ dat;
    dat.X=accelApLF;
    dat.Y=accelMlLF;
    dat.Z=accelVLF;
    return dat;
}
static double JerkAPf ;
static double JerkMLf ;
static double JerkVf ;
static double JerkTf ;

static double accX;
static double accY;
static double accZ;
static unsigned short pointCount=0;
void ACCELPpEquilibreProcessing( _Q16DataXYZ accelFiltree )
{
    _Q16 ACCELRealSamplingFreqQ16 = _Q16ftoi(ACCELRealSamplingFreq);
    // Valeurs utiles de Jerk
    _Q16 JerkAP = _Q16mac((accelFiltree.X +_Q16neg(APF_1)) ,  ACCELRealSamplingFreqQ16,NULLACCU);
    _Q16 JerkML = _Q16mac((accelFiltree.Y + _Q16neg(MLF_1)), ACCELRealSamplingFreqQ16,NULLACCU);
    _Q16 JerkV = _Q16mac((accelFiltree.Z +_Q16neg(VF_1)) , ACCELRealSamplingFreqQ16,NULLACCU);
    _Q16 JerkT = _Q16power(_Q16mac(JerkAP , JerkAP, _Q16mac( JerkML, JerkML,_Q16mac(  JerkV , JerkV,NULLACCU))),_Q16_CONST_0_5F);

   JerkAPf =  _itofQ16(JerkAP);
   JerkMLf =_itofQ16(JerkML);
   JerkVf =_itofQ16(JerkV);
   JerkTf =_itofQ16(JerkT);

   accX = _itofQ16(accelFiltree.X);
   accY = _itofQ16(accelFiltree.Y);
   accZ = _itofQ16(accelFiltree.Z);
   
#ifdef EQUI_IN_DBL
    //if (i > Tau) // "Si filtres bien en place"
    {
        // Calcul des valeurs intermdiaires
        RMSMedioLateralVerticalPhase +=accY * accY+( accZ * accZ);
        SPML += ABS(JerkMLf);
        SPV += ABS(JerkVf);
        SPAPML += sqrt((JerkMLf * JerkMLf) + ( JerkAPf*JerkAPf));
        SPMLV += sqrt((JerkMLf * JerkMLf) + ( JerkVf*JerkVf));
        SPT += JerkTf;
        SAAML += ABS(accY);
        SAAV += ABS(accZ);
        SAAAPML += sqrt((accY * accY)+( accX * accX));
        SAAMLV += sqrt((accY * accY)+( accZ * accZ));
        SAAT += sqrt((accY * accY)+( accZ * accZ)+ ( accX * accX));
        MeanJerkTotalPhase += JerkTf;
        RMSJerkVerticalPhase += JerkVf*JerkVf;
        pointCount++;
    }
#else
   // Calcul des valeurs intermdiaires
    RMSMedioLateralVerticalPhase += _Q16mac(accelFiltree.Y , accelFiltree.Y, _Q16mac( accelFiltree.Z , accelFiltree.Z,NULLACCU));
    SPML += ABS_Q16(JerkML);
    SPV += ABS_Q16(JerkV);
    SPAPML += _Q16norm(_Q16mac(JerkML , JerkML,_Q16mac( JerkAP,JerkAP,NULLACCU)));
    SPMLV += _Q16norm(_Q16mac(JerkML , JerkML,_Q16mac( JerkV,JerkV,NULLACCU)));
    SPT += JerkT;
    SAAML += ABS_Q16(accelFiltree.Y);
    SAAV += ABS_Q16(accelFiltree.Z);
    SAAAPML += _Q16norm(_Q16mac(accelFiltree.Y , accelFiltree.Y ,_Q16mac( accelFiltree.X , accelFiltree.X,NULLACCU)));
    SAAMLV += _Q16norm(_Q16mac(accelFiltree.Y , accelFiltree.Y ,_Q16mac( accelFiltree.Z , accelFiltree.Z,NULLACCU)));
    SAAT += _Q16norm(_Q16mac(accelFiltree.Y , accelFiltree.Y ,_Q16mac( accelFiltree.Z , accelFiltree.Z,_Q16mac( accelFiltree.X , accelFiltree.X,NULLACCU))));
    MeanJerkTotalPhase += JerkT;
    RMSJerkVerticalPhase += _Q16mac(JerkV,JerkV, NULLACCU);
#endif
    APF_1 = accelFiltree.X;
    MLF_1 = accelFiltree.Y;
    VF_1 = accelFiltree.Z;
}

void ACCELPpInitFilters_Equilibre(void)
{
    #ifdef USE_FILTRE_EQUI_ORDRE_2
        FiltrePasseHautButterworthOrdre2Init(&filtrePasseHautAnteroPosterieur, ACCELRealSamplingFreq, FILTRE_EQUI_HP_FC);
        FiltrePasseHautButterworthOrdre2Init(&filtrePasseHautMedioLateral, ACCELRealSamplingFreq, FILTRE_EQUI_HP_FC);
        FiltrePasseHautButterworthOrdre2Init(&filtrePasseHautVertical, ACCELRealSamplingFreq, FILTRE_EQUI_HP_FC);

        FiltrePasseBasButterworthOrdre2Init(&filtrePasseBasAnteroPosterieur, ACCELRealSamplingFreq, FILTRE_EQUI_LP_FC);
        FiltrePasseBasButterworthOrdre2Init(&filtrePasseBasMedioLateral, ACCELRealSamplingFreq, FILTRE_EQUI_LP_FC);
        FiltrePasseBasButterworthOrdre2Init(&filtrePasseBasVertical, ACCELRealSamplingFreq, FILTRE_EQUI_LP_FC);
    #else
        //HP
        FiltrePasseHautButterworthOrdre1Init(&filtrePasseHautAnteroPosterieur, ACCELRealSamplingFreq, FILTRE_EQUI_HP_FC);
        FiltrePasseHautButterworthOrdre1Init(&filtrePasseHautMedioLateral, ACCELRealSamplingFreq, FILTRE_EQUI_HP_FC);
        FiltrePasseHautButterworthOrdre1Init(&filtrePasseHautVertical, ACCELRealSamplingFreq, FILTRE_EQUI_HP_FC);

        FiltrePasseHautButterworthOrdre1Init(&filtrePasseHautAnteroPosterieur_2, ACCELRealSamplingFreq, FILTRE_EQUI_HP_FC);
        FiltrePasseHautButterworthOrdre1Init(&filtrePasseHautMedioLateral_2, ACCELRealSamplingFreq, FILTRE_EQUI_HP_FC);
        FiltrePasseHautButterworthOrdre1Init(&filtrePasseHautVertical_2, ACCELRealSamplingFreq, FILTRE_EQUI_HP_FC);

        //LP
        FiltrePasseBasButterworthOrdre1Init(&filtrePasseBasAnteroPosterieur, ACCELRealSamplingFreq, FILTRE_EQUI_LP_FC);
        FiltrePasseBasButterworthOrdre1Init(&filtrePasseBasMedioLateral, ACCELRealSamplingFreq, FILTRE_EQUI_LP_FC);
        FiltrePasseBasButterworthOrdre1Init(&filtrePasseBasVertical, ACCELRealSamplingFreq, FILTRE_EQUI_LP_FC);

        FiltrePasseBasButterworthOrdre1Init(&filtrePasseBasAnteroPosterieur_2, ACCELRealSamplingFreq, FILTRE_EQUI_LP_FC);
        FiltrePasseBasButterworthOrdre1Init(&filtrePasseBasMedioLateral_2, ACCELRealSamplingFreq, FILTRE_EQUI_LP_FC);
        FiltrePasseBasButterworthOrdre1Init(&filtrePasseBasVertical_2, ACCELRealSamplingFreq, FILTRE_EQUI_LP_FC);
    #endif
    APF_1 = 0;
    MLF_1 = 0;
    VF_1 = 0;

    SPML = 0;
    SPV = 0;
    SPAPML = 0;
    SPMLV = 0;
    SPT = 0;
    SAAML = 0;
    SAAV = 0;
    SAAAPML = 0;
    SAAMLV = 0;
    SAAT = 0;
    pointCount=0;
}

void ACCELPpFilters_EquilibreClear(void)
{
    APF_1 = 0;
    MLF_1 = 0;
    VF_1 = 0;

    SPML = 0;
    SPV = 0;
    SPAPML = 0;
    SPMLV = 0;
    SPT = 0;
    SAAML = 0;
    SAAV = 0;
    SAAAPML = 0;
    SAAMLV = 0;
    SAAT = 0;
    pointCount=0;

    RMSMedioLateralVerticalPhase = 0;
    MeanJerkTotalPhase = 0;
    RMSJerkVerticalPhase = 0;
}

void SendTrameEquilibreStats(void)
{
    unsigned char trameEquilibre[28],pos=0;
    #ifdef EQUI_IN_DBL
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)(SPML));       
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)(SPML));       
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)(SPV));        
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)(SPV));        
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)(SPAPML));     
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)(SPAPML));     
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)(SPMLV));
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)(SPMLV));
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)(SPT));        
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)(SPT));        
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)(SAAML*10));
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)(SAAML*10));
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)(SAAV*10));
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)(SAAV*10));
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)(SAAAPML*10));
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)(SAAAPML*10));
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)(SAAMLV*10));
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)(SAAMLV*10));
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)(SAAT*10));
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)(SAAT*10));
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)(RMSMedioLateralVerticalPhase*10000));
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)(RMSMedioLateralVerticalPhase*10000));
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)(MeanJerkTotalPhase));
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)(MeanJerkTotalPhase));
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)(RMSJerkVerticalPhase));
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)(RMSJerkVerticalPhase));
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)(pointCount));
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)(pointCount));
    #else
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)_itofQ16(SPML));       //0
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)_itofQ16(SPML));       //0
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)_itofQ16(SPV));        //0
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)_itofQ16(SPV));        //0
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)_itofQ16(SPAPML));     //0
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)_itofQ16(SPAPML));     //115
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)_itofQ16(SPMLV));      //0
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)_itofQ16(SPMLV));      //116
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)_itofQ16(SPT));        //128
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)_itofQ16(SPT));        //86
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)_itofQ16(_Q16mac(SAAML,_Q16_CONST_10,NULLACCU)));      //0
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)_itofQ16(_Q16mac(SAAML,_Q16_CONST_10,NULLACCU)));      //128
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)_itofQ16(_Q16mac(SAAV,_Q16_CONST_10,NULLACCU)));       //0
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)_itofQ16(_Q16mac(SAAV,_Q16_CONST_10,NULLACCU)));       //128
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)_itofQ16(_Q16mac(SAAAPML,_Q16_CONST_10,NULLACCU)));    //0
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)_itofQ16(_Q16mac(SAAAPML,_Q16_CONST_10,NULLACCU)));    //128
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)_itofQ16(_Q16mac(SAAMLV,_Q16_CONST_10,NULLACCU)));
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)_itofQ16(_Q16mac(SAAMLV,_Q16_CONST_10,NULLACCU)));
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)_itofQ16(_Q16mac(SAAT,_Q16_CONST_10,NULLACCU)));
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)_itofQ16(_Q16mac(SAAT,_Q16_CONST_10,NULLACCU)));
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)_itofQ16(_Q16mac(RMSMedioLateralVerticalPhase,_Q16_CONST_100,NULLACCU)));
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)_itofQ16(_Q16mac(RMSMedioLateralVerticalPhase,_Q16_CONST_100,NULLACCU)));
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)_itofQ16(MeanJerkTotalPhase));
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)_itofQ16(MeanJerkTotalPhase));
    trameEquilibre[pos++]=LSB_UINT16((unsigned short)_itofQ16(RMSJerkVerticalPhase));
    trameEquilibre[pos++]=MSB_UINT16((unsigned short)_itofQ16(RMSJerkVerticalPhase));
    #endif//EQUI_IN_DBL

    SendMessageToRadio(COORD_SHORT_ADDRESS, TRAME_INFO_EQUILIBRE, pos, trameEquilibre);
}
#endif
#endif